package sample;

import java.lang.Object.*;
import java.io.IOException;
import java.io.FileReader;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import com.toshiba.mwcloud.gs.GridStore;
import com.toshiba.mwcloud.gs.GridStoreFactory;
import com.toshiba.mwcloud.gs.Geometry;
import com.toshiba.mwcloud.gs.GeometryOperator;
import com.toshiba.mwcloud.gs.TimestampUtils;
import com.toshiba.mwcloud.gs.TimeUnit;
import com.toshiba.mwcloud.gs.GSException;
import com.toshiba.mwcloud.gs.Collection;
import com.toshiba.mwcloud.gs.Query;
import com.toshiba.mwcloud.gs.IndexType;
import com.toshiba.mwcloud.gs.RowSet;

import au.com.bytecode.opencsv.CSVReader;

import sample.row.StreetCamera;

import sample.logic.BoundingOption;
import sample.logic.Dimension;
import sample.logic.GeometryLogic;

public class StreetLightApplication {
	private static final String COLLECTION_NAME = "TrafficApplication_101";
	private static final String COLUMN = "coordinates";
	private static final String DIRECTORY = "../data/";
	private static final String CSV_FILE = "StreetLights.csv";
	private static final String SHAPE_PREFIX = "POINT(";


	private static void readCSV(Collection<String,StreetCamera> collection, String filename) throws GSException, IOException {
		CSVReader reader = null;
		String nextLine[];

		try {
			reader = new CSVReader(new FileReader(DIRECTORY + filename));
			reader.readNext();

			while((nextLine = reader.readNext()) != null){
				StreetCamera streetCamera = new StreetCamera();

				streetCamera.cameraId = nextLine[0];
				String latitude = nextLine[1];
				String longitude = nextLine[2];
				String elevation = nextLine[3];

				String wkt = String.format("%s %s %s %s)",SHAPE_PREFIX,latitude,longitude,elevation);

				streetCamera.coordinates = Geometry.valueOf(wkt);

				streetCamera.installation = TimestampUtils.current();
				streetCamera.carsSeen = Integer.parseInt(nextLine[4]);
				streetCamera.violations = Integer.parseInt(nextLine[5]);

				System.out.println("Inserting camera " + streetCamera.cameraId + " into collection");
				collection.put(streetCamera.cameraId,streetCamera);
			}
		
		} catch(GSException e){
			e.printStackTrace();
			System.out.println("Error inserting record into GridDB");
		} catch(IOException e){
			e.printStackTrace();
			System.out.println("Error handling " + filename);
		} finally {
			if(reader != null){
				reader.close();
			}
		}
	}

	public static void main(String args[]) throws GSException, IOException {
		Properties properties = new Properties();
		properties.setProperty("notificationAddress",args[0]);
		properties.setProperty("notificationPort",args[1]);
		properties.setProperty("clusterName",args[2]);
		properties.setProperty("user",args[3]);
		properties.setProperty("password",args[4]);

		GridStore gridstore = null;
		gridstore = GridStoreFactory.getInstance().getGridStore(properties);

		Collection<String,StreetCamera> streetCollection = gridstore.putCollection(COLLECTION_NAME,StreetCamera.class);
		streetCollection.createIndex(COLUMN,IndexType.SPATIAL);
		readCSV(streetCollection,CSV_FILE);
		System.out.println();
		
		GeometryLogic geometryLogic = new GeometryLogic(streetCollection,COLUMN);

		Geometry point = Geometry.valueOf("POINT(40.98 76.43 1.6)");
		System.out.print("Searching for point: ");
		System.out.println(point);
		geometryLogic.getSearchResults(point);
		
		System.out.println("Testing a query against specific ranges");
		System.out.println();

		geometryLogic.searchXRange(35.50,BoundingOption.GREATER);

		System.out.println();

		geometryLogic.searchZRange(0.8,0.3);

		System.out.println();
		geometryLogic.searchYRange(77,82);


		System.out.println("Testing a query against a geographic area");
		geometryLogic.searchArea(40,79,45,85,true);


		System.out.println("Testing a query against a plane");
		geometryLogic.searchVolume(36,73,0.6,42,77,1.7,true);

		List<Double> planePoints = new ArrayList<Double>();

		planePoints.add(38.23);
		planePoints.add(74.34);
		planePoints.add(0.6);
		planePoints.add(43.11);
		planePoints.add(78.67);
		planePoints.add(1.2);
		planePoints.add(40.98);
		planePoints.add(76.43);
		planePoints.add(1.6);

		System.out.println();
		geometryLogic.searchPlane(planePoints,Dimension.THREE_DIMENSIONAL,true);

		System.out.println("Testing Queries against QUADRATICSURFACES");
		double matrix[] = {0.8,0,0,0,0,0.8,0,0,0,0,0.8,0,0,0,0,-1};
		geometryLogic.searchQuadraticSurface(matrix,true);
		System.out.println();

		double ellipticCone[] = {1,0,0,0,1,0,0,0,0.5,0,0,0,-1,0,0,0};
		geometryLogic.searchQuadraticSurface(ellipticCone,true);
		System.out.println();
		
		double cylinder[] = {0.33,0,0,0,0.33,0,0,0,0,0,0,0,-0.12,0,0,0};
		geometryLogic.searchQuadraticSurface(cylinder,true);
		System.out.println();

		String wkt = "POLYHEDRALSURFACE(((43 78 1.0, 44 78 1.0, 44 80 1.0, 43 80 1.0, 43 78 1.0)), ((43 78 1.0, 44 78 1.0, 43.5 79 1.2, 43 78 1.0)), ((44 78 1.0, 43 80 1.0, 43.5 79 1.2, 44 80 1.0)), ((43 80 1.0, 43 78 1.0, 43.5 79 1.2, 43 80 1.0)))";
		Geometry pyramid = Geometry.valueOf(wkt);

		Query<StreetCamera> query = streetCollection.query(COLUMN,pyramid,GeometryOperator.INTERSECT);
		RowSet<StreetCamera> rowSet = query.fetch(false);

		System.out.println("Testing intersection query against a pyramid geometry");
		while(rowSet.hasNext()){
			StreetCamera camera = rowSet.next();
			System.out.println(camera);
		}

		gridstore.dropCollection(COLLECTION_NAME);

		if(gridstore != null){
			gridstore.close();
		}
	}
}
