#include "gridstore.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define COLLECTION_NAME "GeometryCollection243"

typedef struct {
	const GSChar* name;
	double perimeter;
	double volume;
	GSTimestamp creation;
	const GSChar* geom;
} Shape;

GS_STRUCT_BINDING(Shape,
	GS_STRUCT_BINDING_KEY(name,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(perimeter,GS_TYPE_DOUBLE)
	GS_STRUCT_BINDING_ELEMENT(volume,GS_TYPE_DOUBLE)
	GS_STRUCT_BINDING_ELEMENT(creation,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(geom,GS_TYPE_GEOMETRY));

void outputShape(Shape shape){
	GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
	gsFormatTime(shape.creation,timeStr,sizeof(timeStr));

	printf("Name = %s",shape.name);
	printf(" Perimeter = %lf",shape.perimeter);
	printf(" Volume = %lf",shape.volume);
	printf(" Creation = %s",timeStr);
	printf(" Geometry = %s\n",shape.geom);
}


void sample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSCollection* collection;

	const GSPropertyEntry properties[] = {
		{"notificationHost",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};

	size_t propertySize = sizeof(properties) / sizeof(*properties);
	gsGetGridStore(gsGetDefaultFactory(),properties,propertySize,&gridstore);

	gsPutCollection(gridstore,COLLECTION_NAME,GS_GET_STRUCT_BINDING(Shape),NULL,GS_FALSE,&collection);

	gsCreateIndex(collection,"name",GS_INDEX_FLAG_HASH);
	gsCreateIndex(collection,"creation",GS_INDEX_FLAG_TREE);
	gsCreateIndex(collection,"geom",GS_INDEX_FLAG_SPATIAL);

	printf("Successfully created all three index types on %s\n",COLLECTION_NAME);

	Shape point2D;
	point2D.name = "Point2D_101";
	point2D.perimeter = 0;
	point2D.volume = 0;
	point2D.creation = gsCurrentTime();
	point2D.geom = "POINT(1 3.4)";

	gsPutRow(collection,NULL,&point2D,NULL);

	Shape point3D;
	point3D.name = "Point3D_101";
	point3D.perimeter = 0;
	point3D.volume = 0;
	point3D.creation = gsCurrentTime();
	point3D.geom = "POINT(2 -3 0.43)";

	gsPutRow(collection,NULL,&point3D,NULL);

	printf("Successfully inserted 2D and 3D points into %s\n",COLLECTION_NAME);

	Shape line2D;
	line2D.name = "Line2D_101";
	line2D.perimeter = sqrt(2);
	line2D.volume = 0;
	line2D.creation = gsCurrentTime();
	line2D.geom = "LINESTRING(0 0, 1 1)";

	gsPutRow(collection,NULL,&line2D,NULL);

	Shape line3D;
	line3D.name = "Line3D_101";
	line3D.perimeter = (sqrt(12) + sqrt(3));
	line3D.volume = 0;
	line3D.creation= gsCurrentTime();
	line3D.geom = "LINESTRING(0 0 0, 2 2 2, 3 1 1)";

	gsPutRow(collection,NULL,&line3D,NULL);

	printf("Successfully inserted 2D and 3D lines into %s\n",COLLECTION_NAME);

	Shape polygon;
	polygon.name = "Polygon2D_101";
	polygon.perimeter = 4;
	polygon.volume = 1;
	polygon.creation = gsCurrentTime();
	polygon.geom = "POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))";

	gsPutRow(collection,NULL,&polygon,NULL);

	Shape threedSurface;
	threedSurface.name = "Polygon3D_101";
	threedSurface.perimeter = (2 + (2 * (sqrt(2))));
	threedSurface.volume = sqrt(2);
	threedSurface.creation = gsCurrentTime();
	threedSurface.geom = "POLYGON((0 0 0,1 0 1,1 1 1,0 1 0,0 0 0))";

	gsPutRow(collection,NULL,&threedSurface,NULL);

	printf("Successfully inserted 2D and 3D polygons into %s\n",COLLECTION_NAME);

	Shape islandSurface;
	islandSurface.name = "PolygonIsland_101";
	islandSurface.perimeter = 16;
	islandSurface.volume = 12;
	islandSurface.creation = gsCurrentTime();
	islandSurface.geom = "POLYGON((-2 -2 -2, 2 -2 2, 2 2 2, -2 2 2, -2 -2 -2), (-1 -1 -1, 0 -1 0, 0 0 0, -1 0 0, -1 -1 -1))";

	gsPutRow(collection,NULL,&islandSurface,NULL);

	printf("Successfully inserted 3D island Polygon into %s\n",COLLECTION_NAME);

	Shape polySurface;
	polySurface.name = "Polysurface_101";
	polySurface.perimeter = 6; // Placeholder for surface area
	polySurface.volume = 1;
	polySurface.creation = gsCurrentTime();
	polySurface.geom = "POLYHEDRALSURFACE ( ((0 0 0, 0 1 0, 1 1 0, 1 0 0, 0 0 0)),((0 0 0, 0 1 0, 0 1 1, 0 0 1, 0 0 0 )),((0 0 0, 1 0 0, 1 0 1, 0 0 1, 0 0 0)),((1 1 1, 1 0 1, 0 0 1, 0 1 1, 1 1 1)) ,((1 1 1, 1 0 1, 1 0 0, 1 1 0, 1 1 1)),((1 1 1, 1 1 0, 0 1 0, 0 1 1, 1 1 1)) )";

	gsPutRow(collection,NULL,&polySurface,NULL);

	printf("Successfully inserted POLYHEDRALSURFACE into %s\n",COLLECTION_NAME);

	GSQuery* apiQuery;
	GSRowSet* apiRs;

	const GSChar* polygonGeom = "POLYGON((-2 -2 0, 2 -2 0, 2 2 0, -2 2 0, -2 -2 0))";
	gsQueryByGeometry(collection,"geom",polygonGeom,GS_GEOMETRY_OPERATOR_INTERSECT,&apiQuery);
	gsFetch(apiQuery,GS_FALSE,&apiRs);

	printf("\nLooking for rows that spatially intersects %s\n\n",polygonGeom);
	int result = 1;

	Shape apiRow;
	while(gsHasNextRow(apiRs)){
		gsGetNextRow(apiRs,&apiRow);
		printf("API Result %d: ",result++);

		outputShape(apiRow);
	}

	Shape row;
	result = 1;
	GSQuery* query;
	GSRowSet* rowSet;

	const GSChar* tql = "SELECT * WHERE ST_QSFMBRIntersects(ST_MakeQSF(1,0,0,0,1,0,0,0,0,0,0,0,-1,0,0,0),geom) AND creation > TIMESTAMPADD(DAY,NOW(),-1)";
	printf("\nLooking for row that spatially intersect %s\n\n","\'QUADRATICSURFACE(( 1 0 0 0 1 0 0 0 0 0 0 0 -1 0 0 0 ))\'");

	gsQuery(collection,tql,&query);
	gsFetch(query,GS_FALSE,&rowSet);

	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&row);
		printf("TQL Result %d: ",result++);

		outputShape(row);
	}

	gsDropCollection(gridstore,COLLECTION_NAME);
	gsCloseGridStore(&gridstore,GS_TRUE);
}


int main(int argc, char** argv){
	if(argc < 6){
		printf("Need to enter GridDB host, port, cluster name, username, and password\n");
		return 1;
	} else{
		sample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	}
	return 0;
}