#include "gridstore.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "csvparser.h"

#define SHAPE_PREFIX "POLYGON(("
#define PREFIX "../data/img/"
#define PREFIX_LENGTH 12
#define CSV_FILE "../data/CameraReadings.csv"


// The MIT License (MIT)

// Copyright (c) 2015 

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.


typedef struct {
	GSTimestamp timestamp; // Time-type rowkey
	const GSChar* id; // ID of the camera in the database
	const GSChar* url; // Url of the image file
	const GSChar* area; // Spatial area the image encompasses
	uint64_t cars; // Number of cars recorded
	double elevation; // Relative elevation of the image to ground
	GSBlob image; // Actual image file
} TrafficCamera;


// Establish column schema from TrafficCamera Struct
GS_STRUCT_BINDING(TrafficCamera,
	GS_STRUCT_BINDING_KEY(timestamp,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(id,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(url,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(area,GS_TYPE_GEOMETRY)
	GS_STRUCT_BINDING_ELEMENT(cars,GS_TYPE_INTEGER)
	GS_STRUCT_BINDING_ELEMENT(elevation,GS_TYPE_DOUBLE)
	GS_STRUCT_BINDING_ELEMENT(image,GS_TYPE_BLOB));


GSBlob formBlob(const GSChar* fileUrl){
	GSBlob blob; // Initial, default empty blob object
	blob.size = 0; // Default size
	blob.data = NULL; // Default data

	FILE* file; // Image file
	unsigned long filelen; // Image file size
	unsigned char* file_data; // Image file data (in binary data)

	file = fopen(fileUrl,"rb"); // Open file and read its binary data
	if(file == NULL){
		printf("File Not Found!\n"); // Display error message if file not found
	} else {
		fseek(file,0,SEEK_END);
		filelen = ftell(file); // Calculate and find file's length/size
		fseek(file,0,SEEK_SET); // Rewind to beginning of file

		file_data = (char*) malloc(filelen); // Allocate data for blob's data values based on image's size
		if(file_data == NULL){
			printf("Memory Error\n"); // Display error message if data could not be allocated
		} else{
			int read = 1;
			char byte;

			// Read all bytes of the file
			fread(file_data,filelen,sizeof(unsigned char), file);

			blob.size = filelen; // Set blob's size as the size of the image file
			blob.data = file_data; // Have blob's data correspond to byte data of the image file


			printf("File Size: %d\n",blob.size);
		}
		fclose(file); // Clean up and close file
	}
	return blob;
}



void readCsv(GSTimeSeries* timeseries, char* filename){
	CsvParser* parser = CsvParser_new(filename,",",1); // Create csv parser for certain csv file
	CsvRow* header;
	CsvRow* row;

	header = CsvParser_getHeader(parser); // Get header fields of csv file
	if(header == NULL){
		printf("%s\n",CsvParser_getErrorMessage(parser)); // If file empty or is not found 
		exit(1); // exit program
	}

	char** headerFields = CsvParser_getFields(header);
	printf("Fields of csv file: %s\n",filename); 

	int i;
	for(i = 0; i < CsvParser_getNumFields(header); i++){
		printf("Category: %s\n",headerFields[i]); // Display all header fields of the csv file
	}

	while((row = CsvParser_getRow(parser))){
		char** rowFields = CsvParser_getFields(row); // Get line of csv file
		TrafficCamera camera; // Create row-object
		camera.timestamp = gsCurrentTime(); // Set timestamp of timeseries row
		camera.id = rowFields[0]; // Set id of row with csv file line-field

		// Allocate space needed to get file url to fetch image
		char* url = (char *) malloc(sizeof(char) * (strlen(rowFields[1])) + PREFIX_LENGTH);
		
		strcpy(url,PREFIX);
		strcat(url,rowFields[1]);
		
		camera.url = url; // Set image-url of the row

		int coordinateLen = 0;
		
		// Get amount of space needed to allocate for POLYGON wkt-string
		for(i = 2; i < 7; i++){
			coordinateLen += (strlen(rowFields[i]) + 1);
		}

		char* area = (char *) malloc(sizeof(char) * (strlen(SHAPE_PREFIX) + 2 + coordinateLen));
		strcpy(area,SHAPE_PREFIX);

		for(i = 2; i < 6; i++){
			strcat(area,rowFields[i]); // Add corner coordinate to POLYGON
			strcat(area,",");
		}

		strcat(area,rowFields[2]); // Close polygon
		strcat(area,"))"); 

		camera.area = area; // Set 'area' geometry value with WKT POLYGON String

		sscanf(rowFields[6],"%d",&(camera.cars)); // Get cars 'integer' value from csv line
		sscanf(rowFields[7],"%lf",&(camera.elevation)); // Get elevation value from csv line

		GSBlob blob; // Establish initial empty Blob value
		blob = formBlob(camera.url); // See if Blob value can be set from the camera's image url
		camera.image= blob; // Set image blob

		gsPutRow(timeseries,NULL,&camera,NULL); // Insert row into Timeseries
		CsvParser_destroy_row(row);
	}

	CsvParser_destroy(parser);
}

void sample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSTimeSeries* timeseries;

	GSPropertyEntry properties[] = {
		{"notificationAddress",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};
	size_t propertySize = sizeof(properties) / sizeof(*properties);
	gsGetGridStore(gsGetDefaultFactory(),properties,propertySize,&gridstore); // Create and establish connection to GridDB

	// Create Timeseries with TrafficCamera Column Schema and insert into Timeseries
	gsPutTimeSeries(gridstore,"CameraTs123",GS_GET_STRUCT_BINDING(TrafficCamera),NULL,GS_FALSE,&timeseries);

	// Read CSV File line by line and convert lines into rows that can be inserted into the Timeseries
	readCsv(timeseries,CSV_FILE);

	// Perform spatial intersection query filtering for rows with 'cars' values < 100
	const GSChar* tql = "SELECT * WHERE ST_MBRIntersects(area,ST_GeomFromText('POLYGON((8 8, 9 8, 9 9, 8 9, 8 8))')) AND cars < 100";

	GSQuery* query;
	GSRowSet* rowSet;
	TrafficCamera camera;

	// Create, issue, and fetch query
	gsQuery(timeseries,tql,&query);
	gsFetch(query,GS_FALSE,&rowSet);

	printf("Executing TQL Query: %s\n",tql);
	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&camera);
		GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
		gsFormatTime(camera.timestamp,timeStr,sizeof(timeStr)); // Format timestamp into datetime-like string

		printf("Timestamp: %s\n",timeStr); // Display formatted timestamp
		printf("Id: %s",camera.id); // Display id
		printf(" Url: %s",camera.url); // Display image url
		printf(" Area: %s",camera.area); // Display Geometry WKT
		printf(" Cars: %d",camera.cars); // Display Cars
		printf(" Elevation: %lf",camera.elevation); // Display elevation
		printf(" Image Size: %d\n",camera.image.size); // Display image file-size

	}

	gsDropTimeSeries(gridstore,"CameraTs123");
	gsCloseGridStore(&gridstore,GS_TRUE);
}


int main(int argc, char** argv){
	if(argc < 6){
		printf("Need to enter host, port, cluster name, username, password\n");
		return 1;
	}
	else
		sample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	return 0;
}