#include "gridstore.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "csvparser.h"

#define SHAPE_PREFIX "LINESTRING("
#define CSV_FILE "../data/PipeData.csv"


// The MIT License (MIT)

// Copyright (c) 2015 

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.


typedef struct {
	const GSChar* id; // ID of sensor / monitor 
	const GSChar* pipe; // Vector or Linestring value, is GEOMETRY TYPE
	GSTimestamp installation; // Installation time of monitor
	double radius; // Radius of the pipe
	double elevation; // Elevation of the pipe relative to sea level or ground level, 'negative' means underground
} PipeMonitor;


GS_STRUCT_BINDING(PipeMonitor,
	GS_STRUCT_BINDING_KEY(id,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(pipe,GS_TYPE_GEOMETRY)
	GS_STRUCT_BINDING_ELEMENT(installation,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(radius,GS_TYPE_DOUBLE)
	GS_STRUCT_BINDING_ELEMENT(elevation,GS_TYPE_DOUBLE));

char* translateLine(char* line){
	int i;
	// Replace all instances of ':' with ',' 
	// so that a LINESTRING WKT geometry can be formed
	for(i = 0; i < strlen(line); i++){
		if(line[i] == ':')
			line[i] = ',';
	}

	return line;
}

void outputMonitor(PipeMonitor pipeMonitor){
	GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
	gsFormatTime(pipeMonitor.installation,timeStr,sizeof(timeStr));

	printf("ID: %s",pipeMonitor.id);
	printf(" PIPE: %s",pipeMonitor.pipe);
	printf(" INSTALLATION: %s",timeStr);
	printf(" RADIUS: %lf",pipeMonitor.radius);
	printf(" ELEVATION: %lf\n",pipeMonitor.elevation);
}

void readCsv(GSCollection* collection, char* fileName){
	CsvParser* parser = CsvParser_new(fileName,",",1);
	CsvRow* header;
	CsvRow* row;
	int i;

	header = CsvParser_getHeader(parser);
	if(header == NULL){
		printf("%s\n",CsvParser_getErrorMessage(parser)); // If file is empty or not found, exit program
		exit(-1);
	}

	char** headerFields = CsvParser_getFields(header);
	printf("Header Fields of file: %s\n",fileName); // Display CSV Headers

	for(i = 0; i < CsvParser_getNumFields(header); i++){
		printf("Category %d: %s\n",i,headerFields[i]); 
	}

	while((row = CsvParser_getRow(parser))){
		char** rowFields = CsvParser_getFields(row); // Get data values for rows line by line
		PipeMonitor pipeMonitor; // Create row object

		pipeMonitor.id = rowFields[0]; // Set 'id' column of row

		// Allocate space for the length of string needed to set Geometry column
		char* lineString = (char *) malloc(sizeof(char) * (strlen(SHAPE_PREFIX) + 3 + strlen(rowFields[1])));

		strcpy(lineString,SHAPE_PREFIX);
		strcat(lineString,translateLine(rowFields[1]));
		strcat(lineString,")");
		// Create WKT String from row field in CSV line

		pipeMonitor.pipe = lineString; // Set 'pipe' Geometry value with WKT String
		
		pipeMonitor.installation = gsCurrentTime(); // Set installation time


		sscanf(rowFields[2],"%lf",&(pipeMonitor.radius)); // Get radius from csv fields
		sscanf(rowFields[3],"%lf",&(pipeMonitor.elevation)); // Get elevation from csv fields


		gsPutRow(collection,NULL,&pipeMonitor,NULL); // Insert row object into GridDB

		CsvParser_destroy_row(row);
	}

	CsvParser_destroy(parser);
}


void sample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSCollection* collection;

	GSPropertyEntry properties[] = {
		{"notificationAddress",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};

	size_t propertySize = sizeof(properties) / sizeof(*properties);

	gsGetGridStore(gsGetDefaultFactory(),properties,propertySize,&gridstore); // Create and establish connection to GridDB

	// Create Collection with PipeMonitor Column Schema and insert it into GridDB
	gsPutCollection(gridstore,"Pipes224",GS_GET_STRUCT_BINDING(PipeMonitor),NULL,GS_FALSE,&collection);

	// Read CSV File and converts each line into a row-object to insert into the collection
	readCsv(collection,CSV_FILE);

	GSQuery* apiQuery;
	GSRowSet* apiRowSet;
	const GSChar* line = "LINESTRING(1 1 1, 1 0 1, 1 0 0)"; // Geometry to search for with the C API

	gsQueryByGeometry(collection,"pipe",line,GS_GEOMETRY_OPERATOR_INTERSECT,&apiQuery); // Issue API Geometry Query by Intersection
	gsFetch(apiQuery,GS_FALSE,&apiRowSet);

	// Issue and fetch query
	PipeMonitor pipeRow;
	printf("Executing API Query for intersection of %s\n\n",line);
	while(gsHasNextRow(apiRowSet)){
		gsGetNextRow(apiRowSet,&pipeRow);
		// Obtain every row and output its values
		outputMonitor(pipeRow);
	}


	GSQuery* query;
	GSRowSet* rowSet;

	// Issue TQL for Spatial intersection and where elevation > -5.3
	const GSChar* tql = "SELECT * WHERE ST_MBRIntersects(pipe,ST_GeomFromText('LINESTRING(7 2 0, 7 2 4)')) AND elevation > -5.3";


	gsQuery(collection,tql,&query);
	gsFetch(query,GS_FALSE,&rowSet);

	// Create, issue, and fetch TQL Query
	printf("\nExecuting TQL Query: %s\n\n",tql);
	PipeMonitor pipeMonitor;

	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&pipeMonitor);
		
		// Obtain every row and output its values	
		outputMonitor(pipeMonitor);
	}

	gsDropCollection(gridstore,"Pipes224");
	gsCloseGridStore(&gridstore,GS_TRUE);
}


int main(int argc, char** argv){
	if(argc < 6){
		printf("Need to enter GridDB host, port, cluster name, username, password\n");
		return 1;
	} else{
		sample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	}

	return 0;
}