#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gridstore.h"
#include "csvparser.h"
#include "geometryLogic.h"

#define COLLECTION_NAME "SmartBuildings_104"
#define CSV_FILE "../data/Buildings.csv"
#define GEOCOLUMN "layout"

typedef struct Building {
	const GSChar* buildingId;
	const GSChar* layout;
	double volume;
	GSTimestamp installation;
	const GSChar* name;
} Building;

GS_STRUCT_BINDING(Building,
	GS_STRUCT_BINDING_KEY(buildingId,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(layout,GS_TYPE_GEOMETRY)
	GS_STRUCT_BINDING_ELEMENT(volume,GS_TYPE_DOUBLE)
	GS_STRUCT_BINDING_ELEMENT(installation,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(name,GS_TYPE_STRING));

double getVolume(double minX,double minY,double minZ,double maxX,double maxY,double maxZ){
	double xLength = abs(maxX - minX);
	double yLength = abs(maxY - minY);
	double zLength = abs(maxZ - minZ);

	double volume = xLength * yLength * zLength;
	return volume;
}


void readCsv(GSCollection* collection, char* filename){
	CsvParser* parser = CsvParser_new(filename,",",1);
	CsvRow* header;
	CsvRow* row;
	int i;

	header = CsvParser_getHeader(parser);
	if(header == NULL){
		printf("%s\n",CsvParser_getErrorMessage(parser));
		exit(-1);
	}

	while((row = CsvParser_getRow(parser))){
		Building building;
		char** rowFields = CsvParser_getFields(row);

		building.buildingId = rowFields[0];

		double minX, minY, minZ, maxX, maxY, maxZ, value;

		sscanf(rowFields[1],"%lf",&minX);
		sscanf(rowFields[2],"%lf",&minY);
		sscanf(rowFields[3],"%lf",&minZ);
		sscanf(rowFields[4],"%lf",&maxX);
		sscanf(rowFields[5],"%lf",&maxY);
		sscanf(rowFields[6],"%lf",&maxZ);


		char* wkt = formSurfaceWkt(minX,minY,minZ,maxX,maxY,maxZ);

		building.layout = wkt;
		building.volume = getVolume(minX,minY,minZ,maxX,maxY,maxZ);

		building.installation = gsCurrentTime();
		building.name = rowFields[7];

		gsPutRow(collection,NULL,&building,NULL);
		free(wkt);
		CsvParser_destroy_row(row);
	}

	CsvParser_destroy(parser);
}

void showResults(GSRowSet* rowSet){
	Building building;
	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&building);

		GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
		gsFormatTime(building.installation,timeStr,sizeof(timeStr));
	
		printf("Id: %s,", building.buildingId);
		printf(" Surface: %s,", building.layout);
		printf(" Volume: %f,",building.volume);
		printf(" Installation: %s,",timeStr);
		printf(" Name: %s\n",building.name);
	}
}

void buildingSample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSCollection* collection;

	GSPropertyEntry properties[] = {
		{"notificationAddress",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};

	size_t propCount = sizeof(properties) / sizeof(*properties);
	gsGetGridStore(gsGetDefaultFactory(),properties,propCount,&gridstore);
	if(gridstore == NULL){
		printf("Error\n");
		exit(-1);
	}	

	GSResult result = gsPutCollection(gridstore,"SmartBuildings_104",GS_GET_STRUCT_BINDING(Building),NULL,GS_FALSE,&collection);
	printf("API RESULT: %d\n",result);

	if(collection == NULL || result != GS_RESULT_OK){
		printf("Collection not created!\n");
		exit(-1);
	}
	readCsv(collection,CSV_FILE);

	GSRowSet* rowSet;
	rowSet = searchVolume(collection,GEOCOLUMN,0,5,0,5,0,5,true);
	showResults(rowSet);

	rowSet = searchArea(collection,GEOCOLUMN,10,0,15,12,true);
	showResults(rowSet);

	double matrix[16] = {0.5,0,0,0,0.5,0,0,0,0.5,0,0,0,-1,0,0,0};
	rowSet = searchQuadratic(collection,GEOCOLUMN,matrix,true);
	showResults(rowSet);

	const GSChar* wkt = "POLYHEDRALSURFACE(((3 4 -3, 7 8 -3, 10 11 -3, 3 4 -3)), ((7 8 -3, 10 11 -3, 2 2 -1, 7 8 -3)), ((3 4 -3, 7 8 -3, 2 2 -1, 3 4 -3)), ((3 4 -3, 10 11 -3, 2 2 -1, 3 4 -3)))";
	GSQuery* query;
	GSRowSet* newRs;
	gsQueryByGeometry(collection,GEOCOLUMN,wkt,GS_GEOMETRY_OPERATOR_INTERSECT,&query);
	gsFetch(query,GS_FALSE,&newRs);
	showResults(newRs);


	gsDropCollection(gridstore,"SmartBuildings_104");
	gsCloseGridStore(&gridstore,GS_FALSE);
}

int main(int argc,char** argv){
	if(argc < 6){
		printf("Need to enter GridDB host, port, cluster name, username, and password\n");
		return 1;
	} else {
		buildingSample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	}

	return 0;
}