#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "gridstore.h"
#include "csvparser.h"
#include "geometryLogic.h"

#define COLLECTION_NAME "TrafficApplication_101"
#define GEOCOLUMN "coordinates"
#define CSV_FILE "../data/StreetLights.csv"



typedef struct StreetCamera{
	const GSChar* cameraId;
	const GSChar* coordinates;
	GSTimestamp installation;
	int carsSeen;
	int violations;
} StreetCamera;


GS_STRUCT_BINDING(StreetCamera,
	GS_STRUCT_BINDING_KEY(cameraId,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(coordinates,GS_TYPE_GEOMETRY)
	GS_STRUCT_BINDING_ELEMENT(installation,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(carsSeen,GS_TYPE_INTEGER)
	GS_STRUCT_BINDING_ELEMENT(violations,GS_TYPE_INTEGER));


void readCsv(GSCollection* collection, char* filename){
	CsvParser* parser = CsvParser_new(filename,",",1);
	CsvRow* header;
	CsvRow* row;
	int i;

	header = CsvParser_getHeader(parser);
	if(header == NULL){
		printf("%s\n",CsvParser_getErrorMessage(parser)); // Exit program if file is empty or not found
		exit(-1);
	}

	while((row = CsvParser_getRow(parser))){
		StreetCamera camera;
		char** rowFields = CsvParser_getFields(row);

		camera.cameraId = rowFields[0];
		int pointLength = 0;
		for(i = 1; i < 4; i++){
			pointLength += (strlen(rowFields[i]) + 1);
		}

		char* point = (char*) malloc(sizeof(char) * (strlen(POINT_PREFIX) + 2 + pointLength));
		strcpy(point,POINT_PREFIX);

		for(i = 1;i < 4;i++){
			strcat(point,rowFields[i]);
			strcat(point, " ");
		}

		strcat(point,")");

		camera.coordinates = point;
		camera.installation = gsCurrentTime();

		sscanf(rowFields[4],"%d",&(camera.carsSeen));
		sscanf(rowFields[5],"%d",&(camera.violations));

		gsPutRow(collection,NULL,&camera,NULL);
		free(point);
		CsvParser_destroy_row(row);
	}

	CsvParser_destroy(parser);
}

void showResults(GSRowSet* rowSet){
	StreetCamera camera;
	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&camera);

		GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
		gsFormatTime(camera.installation,timeStr,sizeof(timeStr));

		printf("Id: %s", camera.cameraId);
		printf(" ,Point: %s", camera.coordinates);
		printf(" ,Timestamp: %s",timeStr);
		printf(" ,Cars Seen: %d",camera.carsSeen);
		printf(" ,Violations: %d\n",camera.violations);
	}
}

void streetLightSample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSCollection* collection;

	GSPropertyEntry properties[] = {
		{"notificationAddress",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};

	size_t propCount = sizeof(properties) / sizeof(*properties);

	gsGetGridStore(gsGetDefaultFactory(),properties,propCount,&gridstore);
	gsPutCollection(gridstore,COLLECTION_NAME,GS_GET_STRUCT_BINDING(StreetCamera),NULL,GS_FALSE,&collection);

	readCsv(collection,CSV_FILE);

	const GSChar* point =  "POINT(47.33 84.72 0.7)";
	GSRowSet* rowSet;
	StreetCamera camera;

	printf("Searching for %s\n",point);
	rowSet = executeSearch(collection,GEOCOLUMN,point);
	showResults(rowSet);

	char* wkt = searchXRange_1(35.5,GREATER);
	char* tql = formTql(GEOCOLUMN,wkt,true);

	rowSet = executeTql(collection,tql);
	showResults(rowSet);
	free(tql);
	free(wkt);


	printf("Performing a spatial query against a 2D geographic area\n\n");
	rowSet = searchArea(collection,GEOCOLUMN,40,79,45,85,true);
	showResults(rowSet);

	printf("Performing a spatial query against a 3D geographic volume\n\n");
	rowSet = searchVolume(collection,GEOCOLUMN,36,73,0.6,42,77,1.7,true);
	showResults(rowSet);

	printf("Performing a spatial query against a 3D geographic area\n\n");
	double plane[9] = {38.23,74.34,0.6,43.11,78.67,1.2,40.98,76.43,1.6};
	rowSet = searchPlane(collection,GEOCOLUMN,plane,9,THREE_DIMENSIONAL,true);
	showResults(rowSet);

	printf("Performing a spatial query against a QUADRATICSURFACE\n\n");
	double matrix[16] = {0.33,0,0,0,0.33,0,0,0,0,0,0,0,-0.12,0,0,0};
	rowSet = searchQuadratic(collection,GEOCOLUMN,matrix,true);
	showResults(rowSet);

	const GSChar* polySurface = "POLYHEDRALSURFACE(((43 78 1.0, 44 78 1.0, 44 80 1.0, 43 80 1.0, 43 78 1.0)), ((43 78 1.0, 44 78 1.0, 43.5 79 1.2, 43 78 1.0)), ((44 78 1.0, 43 80 1.0, 43.5 79 1.2, 44 80 1.0)), ((43 80 1.0, 43 78 1.0, 43.5 79 1.2, 43 80 1.0)))";
	GSQuery* query;
	GSRowSet* newRs;
	gsQueryByGeometry(collection,GEOCOLUMN,polySurface,GS_GEOMETRY_OPERATOR_INTERSECT,&query);
	gsFetch(query,GS_FALSE,&newRs);
	showResults(newRs);


	gsDropCollection(gridstore,COLLECTION_NAME);
	gsCloseGridStore(&gridstore,GS_TRUE);
}




int main(int argc,char** argv){
	if(argc < 6){
		printf("Need to enter GridDB host, port, cluster name, username, and password\n");
		return 1;
	} else {
		streetLightSample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	}

	return 0;
}