package sample;

import java.util.Properties;
import java.util.Date;
import java.lang.Object.*;
import java.util.List;
import java.util.ArrayList;

import java.text.ParseException;
import java.io.IOException;
import java.io.FileReader;

import com.toshiba.mwcloud.gs.Collection;
import com.toshiba.mwcloud.gs.GSException;
import com.toshiba.mwcloud.gs.GridStore;
import com.toshiba.mwcloud.gs.GridStoreFactory;
import com.toshiba.mwcloud.gs.Geometry;
import com.toshiba.mwcloud.gs.GeometryOperator;
import com.toshiba.mwcloud.gs.RowSet;
import com.toshiba.mwcloud.gs.RowKey;
import com.toshiba.mwcloud.gs.Query;
import com.toshiba.mwcloud.gs.IndexType;
import com.toshiba.mwcloud.gs.TimeUnit;
import com.toshiba.mwcloud.gs.TimestampUtils;


public class GeometrySample {

	static class Shape {
		@RowKey String name;
		double perimeter;
		double volume;
		Date creation;
		Geometry geom;

		public String toString(){
			return String.format("Shape: Name = %s, Perimeter = %f, Volume = %f, Creation Date = %s, WKT = %s",name,perimeter,volume,creation.toString(),geom.toString());
		}
	}

	public static void main(String args[]) throws GSException {
		Properties properties = new Properties();
		properties.setProperty("notificationAddress",args[0]);
		properties.setProperty("notificationPort",args[1]);
		properties.setProperty("clusterName",args[2]);
		properties.setProperty("user",args[3]);
		properties.setProperty("password",args[4]);

		GridStore gridstore = null;
		gridstore = GridStoreFactory.getInstance().getGridStore(properties);

		Collection<String,Shape> geometryCollection = gridstore.putCollection("GeometryCollection121",Shape.class);

		// Establish Indexes 
		geometryCollection.createIndex("name",IndexType.HASH);
		geometryCollection.createIndex("creation",IndexType.TREE);
		geometryCollection.createIndex("geom",IndexType.SPATIAL);

		System.out.println("Successfully created indexes on GeometryCollection121");

		Shape point2D = new Shape();
		point2D.name = "Point2D_101";
		point2D.perimeter = 0;
		point2D.volume = 0;
		point2D.creation = TimestampUtils.current();
		point2D.geom = Geometry.valueOf("POINT(1 3.4)");

		geometryCollection.put(point2D.name,point2D);

		Shape point3D = new Shape();
		point3D.name = "Point3D_101";
		point3D.perimeter = 0;
		point3D.volume = 0;
		point3D.creation = TimestampUtils.current();
		point3D.geom = Geometry.valueOf("POINT(2 -3 0.43)");

		geometryCollection.put(point3D.name,point3D);

		System.out.println("Successfully Inserted 2D and 3D points.");

		Shape line2D = new Shape();
		line2D.name = "Line2D_101";
		line2D.perimeter = Math.sqrt(2);
		line2D.volume = 0;
		line2D.creation = TimestampUtils.current();
		line2D.geom = Geometry.valueOf("LINESTRING(0 0, 1 1)");

		geometryCollection.put(line2D.name,line2D);

		Shape line3D = new Shape();
		line3D.name = "Line3D_101";
		line3D.perimeter = Math.sqrt(12) + Math.sqrt(3);
		line3D.volume = 0;
		line3D.creation= TimestampUtils.current();
		line3D.geom = Geometry.valueOf("LINESTRING(0 0 0, 2 2 2, 3 1 1)");

		geometryCollection.put(line3D.name,line3D);

		System.out.println("Successfully inserted 2D and 3D lines");

		Shape polygon = new Shape();
		polygon.name = "Polygon2D_101";
		polygon.perimeter = 4;
		polygon.volume = 1;
		polygon.creation = TimestampUtils.current();
		polygon.geom = Geometry.valueOf("POLYGON((0 0, 1 0, 1 1, 0 1, 0 0))");

		geometryCollection.put(polygon.name,polygon);

		Shape threedSurface = new Shape();
		threedSurface.name = "Polygon3D_101";
		threedSurface.perimeter = 2 + (2 * (Math.sqrt(2)));
		threedSurface.volume = Math.sqrt(2);
		threedSurface.creation = TimestampUtils.current();
		threedSurface.geom = Geometry.valueOf("POLYGON((0 0 0,1 0 1,1 1 1,0 1 0,0 0 0))");

		geometryCollection.put(threedSurface.name,threedSurface);

		System.out.println("Successfully inserted 2D and 3D and Polygons.");

		Shape islandSurface = new Shape();
		islandSurface.name = "PolygonIsland_101";
		islandSurface.perimeter = 16;
		islandSurface.volume = 12;
		islandSurface.creation = TimestampUtils.current();
		islandSurface.geom = Geometry.valueOf("POLYGON((-2 -2 -2, 2 -2 2, 2 2 2, -2 2 2, -2 -2 -2), (-1 -1 -1, 0 -1 0, 0 0 0, -1 0 0, -1 -1 -1))");

		geometryCollection.put(islandSurface.name,islandSurface);

		System.out.println("Successfully inserted Island Polygon");

		Shape polySurface = new Shape();
		polySurface.name = "Polysurface_101";
		polySurface.perimeter = 6; // Placeholder for surface area
		polySurface.volume = 1;
		polySurface.creation = TimestampUtils.current();
		polySurface.geom = Geometry.valueOf("POLYHEDRALSURFACE ( ((0 0 0, 0 1 0, 1 1 0, 1 0 0, 0 0 0))," + 
				"((0 0 0, 0 1 0, 0 1 1, 0 0 1, 0 0 0 ))," + 
				"((0 0 0, 1 0 0, 1 0 1, 0 0 1, 0 0 0))," + 
				"((1 1 1, 1 0 1, 0 0 1, 0 1 1, 1 1 1)) ," + 
				"((1 1 1, 1 0 1, 1 0 0, 1 1 0, 1 1 1))," + 
				"((1 1 1, 1 1 0, 0 1 0, 0 1 1, 1 1 1)) )");

		geometryCollection.put(polySurface.name, polySurface);

		System.out.println("Successfully inserted POLYHEDRALSURFACE");


		Shape quadratic = new Shape();
		quadratic.name = "Quadratic_101";
		quadratic.perimeter = 0;
		quadratic.volume = 0;
		quadratic.creation = TimestampUtils.current();
		quadratic.geom = Geometry.valueOf("QUADRATICSURFACE(( 84 2 3 5 6 7 8 9 10 12 13 14 15 16 17 18 ))");

		System.out.println("Successfully created QUADRATICSURFACE");

		// In the beta version of GridDB SE 2.9 QUADRATIC SURFACES CANNOT BE INSERTED AS COLUMN VALUES
		//geometryCollection.put(quadratic.name, quadratic);


		System.out.println("Looking for Rows whose 'geom' value is is POLYGON((-2 -2 0, 2 -2 0, 2 2 0, -2 2 0, -2 -2 0))");
		System.out.println();

		Query<Shape> generalQuery = geometryCollection.query("geom",Geometry.valueOf("POLYGON((-2 -2 0, 2 -2 0, 2 2 0, -2 2 0, -2 -2 0))"),GeometryOperator.INTERSECT);
		RowSet<Shape> generalRs = generalQuery.fetch(false);

		int result = 1;
		while(generalRs.hasNext()){
			Shape shape = generalRs.next();
			System.out.print(String.format("Result %d: ",result++));
			System.out.println(shape);
		}


		String tql = "SELECT * WHERE ST_QSFMBRIntersects(ST_MakeQSF(1,0,0,0,1,0,0,0,0,0,0,0,-1,0,0,0),geom) AND creation > TIMESTAMPADD(DAY,NOW(),-1)";
		Query<Shape> query = geometryCollection.query(tql);
		RowSet<Shape> rowSet = query.fetch(false);
		result = 1;

		System.out.println();
		System.out.println("Looking for row that spatially intersect");
		System.out.println("'QUADRATICSURFACE(( 1 0 0 0 1 0 0 0 0 0 0 0 -1 0 0 0 ))'");
		System.out.println();


		while(rowSet.hasNext()){
			Shape shape = rowSet.next();
			System.out.print(String.format("TQL Result %d: ",result++));
			System.out.println(shape);
		}

		gridstore.dropCollection("GeometryCollection121");


		if(gridstore != null)
			gridstore.close();
	}
}