package sample.logic;

import java.util.List;
import java.util.ArrayList;

import java.lang.Object.*;

import com.toshiba.mwcloud.gs.Geometry;
import com.toshiba.mwcloud.gs.GeometryOperator;
import com.toshiba.mwcloud.gs.GSException;
import com.toshiba.mwcloud.gs.Collection;
import com.toshiba.mwcloud.gs.TimestampUtils;
import com.toshiba.mwcloud.gs.TimeUnit;
import com.toshiba.mwcloud.gs.Query;
import com.toshiba.mwcloud.gs.RowSet;

import sample.logic.BoundingOption;
import sample.logic.Dimension;

public class GeometryLogic {
	private static final String QUERY_PREFIX = "SELECT * WHERE";
	private static final double MINIMUM = Double.MAX_VALUE * -1;
	private static final double MAXIMUM = Double.MAX_VALUE;
	private static final double MIN_VALUE = Double.MIN_VALUE;

	private double minX, maxX;
	private double minY, maxY;
	private double minZ, maxZ;
	private Collection<String,?> collection;
	private String geometryColumn;


	public GeometryLogic(){
		setInitialRanges();
		collection = null;
		geometryColumn = "";
	}

	public GeometryLogic(Collection<String,?> collection){
		setInitialRanges();
		this.collection = collection;
		geometryColumn = "";
	}

	public GeometryLogic(Collection<String,?> collection, String column){
		setInitialRanges();
		this.collection = collection;
		geometryColumn = column;
	}

	public void setCollection(Collection<String,?> collection){
		this.collection = collection;
	}

	public void setColumn(String column){
		geometryColumn = column;
	}

	private void setInitialRanges(){
		minX = MINIMUM;
		minY = MINIMUM;
		minZ = MINIMUM;
		maxX = MAXIMUM;
		maxY = MAXIMUM;
		maxZ = MAXIMUM;
	}

	public Geometry formPoint(double x, double y){
		String wkt = String.format("POINT(%.9f %.9f)",x,y);
		Geometry point = Geometry.valueOf(wkt);
		return point;
	}

	public Geometry formPoint(double x, double y, double z){
		String wkt = String.format("POINT(%.9f %.9f %.9f)",x,y,z);
		Geometry point = Geometry.valueOf(wkt);
		return point;
	}

	public String formBoundingSquare(double minX, double minY,double maxX, double maxY){
		String bottomLeft = String.format("%.9f %.9f",minX,minY);
		String bottomRight = String.format("%.9f %.9f",maxX,minY);
		String topLeft = String.format("%.9f %.9f",minX,maxY);
		String topRight = String.format("%.9f %.9f",maxX,maxY);

		String wkt = String.format("POLYGON((%s,%s,%s,%s,%s))",bottomLeft,bottomRight,topRight,topLeft,bottomLeft);
		return wkt;
	}

	public String formSurfaceWkt(double minX,double minY,double minZ,double maxX,double maxY,double maxZ){
		String bottomFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",minX,minY,minZ,minX,maxY,minZ,maxX,maxY,minZ,maxX,minY,minZ,minX,minY,minZ);
		String leftFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",minX,minY,minZ,minX,maxY,minZ,minX,maxY,maxZ,minX,minY,maxZ,minX,minY,minZ);
		String frontFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",minX,minY,minZ,maxX,minY,minZ,maxX,minY,maxZ,minX,minY,maxZ,minX,minY,minZ);
		String topFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",maxX,maxY,maxZ,maxX,minY,maxZ,minX,minY,maxZ,minX,maxY,maxZ,maxX,maxY,maxZ);
		String rightFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",maxX,maxY,maxZ,maxX,minY,maxZ,maxX,minY,minZ,maxX,maxY,minZ,maxX,maxY,maxZ);
		String backFace = String.format("((%.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f, %.9f %.9f %.9f))",maxX,maxY,maxZ,maxX,maxY,minZ,minX,maxY,minZ,minX,maxY,maxZ,maxX,maxY,maxZ);

		String wkt = String.format("POLYHEDRALSURFACE(%s,%s,%s,%s,%s,%s)",bottomFace,leftFace,frontFace,topFace,rightFace,backFace);
		return wkt;
	}

	private double formApproximation(double value,BoundingOption option){
		if(option == BoundingOption.LESSER){
			return (value - MIN_VALUE);
		}
		if(option == BoundingOption.GREATER){
			return (value + MIN_VALUE);
		}
		return value;
	}

	private String formPlane(List<Double> points, Dimension dimension){
		String polygon = "POLYGON((";
		int stopIndex = 2;
		if(dimension == Dimension.THREE_DIMENSIONAL){
			stopIndex = 3;
		}

		double value;
		for(int i = 0; i < points.size(); i++){
			value = (double) points.get(i);
			polygon += String.format("%f ",value);

			if((i + 1) % stopIndex == 0  && stopIndex > 0){
				polygon += ",";
			}
		}

		for(int j = 0; j < stopIndex; j++){
			value = (double) points.get(j);
			polygon += String.format("%f ",value);

		}

		polygon += "))";
		return polygon;
	}

	public String formTql(String column,Geometry geometry,boolean include){
		String exclude = include ? "" : "NOT";
		String tql = String.format("%s %s ST_MBRIntersects(%s,ST_GeomFromText('%s'))",QUERY_PREFIX,exclude,column,geometry.toString());
		return tql;
	}

	public String formQSFTql(String column,double matrix[],boolean include){
		int max = 16;
		int stopIndex = matrix.length > 16 ? max : matrix.length;
		
		String wkt = "";
		String exclude = include ? " " : " NOT";
		for(int i = 0; i < max; i++){
			if(i >= stopIndex){
				wkt += ("0");
			} else {
				wkt += (String.valueOf(matrix[i]));
			}

			if(i < max - 1){
				wkt += ",";
			}
		}

		String tql = String.format("%s %s ST_QSFMBRIntersects(ST_MakeQSF(%s),%s)",QUERY_PREFIX,exclude,wkt,column);
		return tql;

	}

	public Geometry formBoundingSurface(Dimension dimension){
		String wkt = "";
		if(dimension == Dimension.THREE_DIMENSIONAL){
			wkt = formSurfaceWkt(minX,minY,minZ,maxX,maxY,maxZ);
		} else if(dimension == Dimension.TWO_DIMENSIONAL){
			wkt = formBoundingSquare(minX,minY,maxX,maxY);
		}

		Geometry geometry = Geometry.valueOf(wkt);
		return geometry;		
	}

	public void getSearchResults(Geometry geometry) throws GSException {
		Query<?> query = collection.query(geometryColumn,geometry,GeometryOperator.INTERSECT);
		RowSet<?> rowSet = query.fetch(false);

		while(rowSet.hasNext()){
			Object row = (Object) rowSet.next();
			System.out.println(row);
		}
	}

	public void executeTql(String tql) throws GSException {
		Query<?> query = collection.query(tql);
		RowSet<?> rowSet = query.fetch(false);

		while(rowSet.hasNext()){
			Object row = (Object) rowSet.next();
			System.out.println(row);
		}
	}

	public void searchArea(double minLatitude, double minLongitude, double maxLatitude, double maxLongitude, boolean include) throws GSException{
		String polygon = formBoundingSquare(minLatitude,minLongitude,maxLatitude,maxLongitude);
		if(include){
			getSearchResults(Geometry.valueOf(polygon));
		} else {
			String tql = formTql(geometryColumn,Geometry.valueOf(polygon),include);
			executeTql(tql);
		}
	}

	public void searchVolume(double minX,double maxX,double minY,double maxY,double minZ,double maxZ,boolean include) throws GSException{
		String box = formSurfaceWkt(minX,minY,minZ,maxX,maxY,maxZ);
		if(include){
			getSearchResults(Geometry.valueOf(box));
 		} else {
 			String tql = formTql(geometryColumn,Geometry.valueOf(box),include);
 			executeTql(tql);
 		}
	}

	public void searchXRange(double xValue, BoundingOption option) throws GSException{
		if(option == BoundingOption.GREATER){
			xValue = formApproximation(xValue,option);
			minX = xValue;
		} else if(option == BoundingOption.LESSER){
			xValue = formApproximation(xValue,option);
			maxX = xValue;
		} else if(option == BoundingOption.GREATEREQ){
			minX  = xValue;
		} else if(option == BoundingOption.LESSEREQ){
			maxX = xValue;
		}

		Geometry box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		System.out.println(box);
		getSearchResults(box);
		setInitialRanges();
	}

	public void searchXRange(double firstX, double secondX) throws GSException {
		Geometry box = Geometry.valueOf("LINESTRING(EMPTY)");

		if(firstX == secondX){
			getSearchResults(box);
			return;
		}

		double min = firstX;
		double max = secondX;

		if(firstX > secondX){
			min = secondX;
			max = firstX;
		}

		minY = min;
		maxY = max;
		box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		getSearchResults(box);

		setInitialRanges();
	}

	public void searchYRange(double yValue, BoundingOption option) throws GSException{
		if(option == BoundingOption.GREATER){
			yValue = formApproximation(yValue,option);
			minY = yValue;
		} else if(option == BoundingOption.LESSER){
			yValue = formApproximation(yValue,option);
			maxY = yValue;
		} else if(option == BoundingOption.GREATEREQ){
			minY = yValue;
		} else if(option == BoundingOption.LESSEREQ){
			maxY = yValue;
		}

		Geometry box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		getSearchResults(box);
		setInitialRanges();
	}

	public void searchYRange(double firstY, double secondY) throws GSException {
		Geometry box = Geometry.valueOf("LINESTRING(EMPTY)");

		if(firstY == secondY){
			getSearchResults(box);
			return;
		}

		double min = firstY;
		double max = secondY;

		if(firstY > secondY){
			min = secondY;
			max = firstY;
		}

		minY = min;
		maxY = max;
		box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		getSearchResults(box);
		setInitialRanges();
	}	
	public void searchZRange(double zValue, BoundingOption option) throws GSException{
		if(option == BoundingOption.GREATER){
			zValue = formApproximation(zValue,option);
			minZ = zValue;
		} else if(option == BoundingOption.LESSER){
			zValue = formApproximation(zValue,option);
			maxZ = zValue;
		} else if(option == BoundingOption.GREATEREQ){
			minZ = zValue;
		} else if(option == BoundingOption.LESSEREQ){
			maxZ = zValue;
		}

		Geometry box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		getSearchResults(box);
		setInitialRanges();
	}

	public void searchZRange(double firstZ, double secondZ) throws GSException {
		Geometry box = Geometry.valueOf("LINESTRING(EMPTY)");

		if(firstZ == secondZ){
			getSearchResults(box);
			return;
		}

		double min = firstZ;
		double max = secondZ;

		if(firstZ > secondZ){
			min = secondZ;
			max = firstZ;
		}


		minZ = min;
		maxZ = max;
		box = formBoundingSurface(Dimension.THREE_DIMENSIONAL);
		getSearchResults(box);
		setInitialRanges();
	}


	public void searchQuadraticSurface(double matrix[],boolean include) throws GSException {
		String tql = formQSFTql(geometryColumn,matrix,include);
		executeTql(tql);
	}

	public void searchPlane(List<Double> points,Dimension dimension,boolean include) throws GSException {
		String planeText = formPlane(points,dimension);
		Geometry plane = Geometry.valueOf(planeText);

		if(include){
			getSearchResults(plane);
		} else {
			String tql = formTql(geometryColumn,plane,include);
			executeTql(tql);
		}
	}
}

