#include "gridstore.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "csvparser.h"

#define SHAPE_PREFIX "POINT("
#define CSV_FILE "../data/trafficLights.csv"

// The MIT License (MIT)

// Copyright (c) 2015 

// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:

// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

typedef struct {
	const GSChar* tag; // Row Key
	const GSChar* point; // Latitude, Longitude, and Elevation of IoT Device
	GSTimestamp installation; // Installation Date
	GSBool active; // Whether the device is in its operating hours
	uint64_t reports; // Number of reports the sensor has made
	const GSChar* streets; // Cross streets of device
}  TrafficLight;


// Create Static Column Schema from TrafficLight Struct
GS_STRUCT_BINDING(TrafficLight,
	GS_STRUCT_BINDING_KEY(tag,GS_TYPE_STRING)
	GS_STRUCT_BINDING_ELEMENT(point,GS_TYPE_GEOMETRY)
	GS_STRUCT_BINDING_ELEMENT(installation,GS_TYPE_TIMESTAMP)
	GS_STRUCT_BINDING_ELEMENT(active,GS_TYPE_BOOL)
	GS_STRUCT_BINDING_ELEMENT(reports,GS_TYPE_INTEGER)
	GS_STRUCT_BINDING_ELEMENT(streets,GS_TYPE_STRING));



void readCsv(GSCollection* collection, char* fileName){
	CsvParser* parser = CsvParser_new(fileName,",",1);
	CsvRow* header;
	CsvRow* row;
	int i;

	header = CsvParser_getHeader(parser);
	if(header == NULL){
		printf("%s\n",CsvParser_getErrorMessage(parser)); // Exit program if file is empty or not found
		exit(-1);
	} 

	char** headerFields = CsvParser_getFields(header);
	printf("Fields of CSV File: %s\n",fileName);

	for(i = 0; i < CsvParser_getNumFields(header); i++){
		printf("Category %d: %s\n",i,headerFields[i]); // Display all the categories of the CSV File
	}

	while((row = CsvParser_getRow(parser))){
		char** rowFields = CsvParser_getFields(row); // Separate all fields of file's line
		TrafficLight trafficLight; // Create row object

		trafficLight.tag = rowFields[0]; // Set row key
		int pointLen = 0;

		for(i = 1; i < 4; i++){
			pointLen += (strlen(rowFields[i]) + 1); // Determine length needed to allocate for Geometry
		}

		char* point = (char *) malloc(sizeof(char) * (strlen(SHAPE_PREFIX) + 1 + pointLen)); // Allocate WKT string
		strcpy(point, SHAPE_PREFIX);

		// Add latitude,longitude, and elevation to POINT string
		for(i = 1; i < 4; i++){
			strcat(point,rowFields[i]);
			strcat(point," ");
		}

		strcat(point,")");

		trafficLight.point = point; // Set installation time, default is current time
		trafficLight.installation = gsCurrentTime();

		// If value of field in the csv file equals true, set to GS_TRUE : GS_FALSE otherwise
		if((strcmp(rowFields[4],"true")) == 0){
			trafficLight.active = GS_TRUE;
		} else{
			trafficLight.active = GS_FALSE;
		}

		sscanf(rowFields[5],"%d",&(trafficLight.reports)); // Get report count
		trafficLight.streets = rowFields[6]; // Get cross streets

		gsPutRow(collection,NULL,&trafficLight,NULL); // Insert Row into collection
		CsvParser_destroy_row(row);
	}

	CsvParser_destroy(parser);
}

void outputTrafficLight(TrafficLight trafficLight, GSChar* timeString){
	printf(" Tag: %s",trafficLight.tag);
	printf(" Point: %s",trafficLight.point);
	printf(" Installation: %s",timeString);
	printf(" Active: %s", trafficLight.active ? "true" : "false");
	printf(" Reports %d",trafficLight.reports);
	printf(" Streets: %s\n",trafficLight.streets);
}

void sample(const char* host, const char* port, const char* clusterName, const char* user, const char* password){
	GSGridStore* gridstore;
	GSCollection* collection;

	GSPropertyEntry properties[] = {
		{"notificationAddress",host},
		{"notificationPort",port},
		{"clusterName",clusterName},
		{"user",user},
		{"password",password}
	};

	size_t propertyCount = sizeof(properties) / sizeof(*properties);
	gsGetGridStore(gsGetDefaultFactory(),properties,propertyCount,&gridstore); // Establish GridDB Connection

	// Insert Collection with TrafficLight schema into GridDB
	gsPutCollection(gridstore,"Lights21",GS_GET_STRUCT_BINDING(TrafficLight),NULL,GS_FALSE,&collection);

	// Read data points from CSV_FILE into rows and insert them into collection
	readCsv(collection,CSV_FILE);

	const GSChar* point = "POINT(47.33 84.72 0.7)"; // Point to search for with C API

	TrafficLight row;
	GSQuery* apiQuery;
	GSRowSet* apiRowSet;


	gsQueryByGeometry(collection,"point",point,GS_GEOMETRY_OPERATOR_INTERSECT,&apiQuery); // Execute Intersection search with C API
	gsFetch(apiQuery,GS_FALSE,&apiRowSet);

	// Create, issue, and fetch queries, and output its resulting rows
	printf("Executing API Search for Intersection on: %s\n\n",point);
	while(gsHasNextRow(apiRowSet)){
		gsGetNextRow(apiRowSet,&row);
		GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
		gsFormatTime(row.installation,timeStr,sizeof(timeStr));

		outputTrafficLight(row,timeStr);
	}


	// Issue a 2D Intersection search with TQL and filter for other conditions
	// Advantage in using TQL
	const GSChar* tql = "SELECT * WHERE ST_MBRIntersects(point,ST_GeomFromText('POLYGON((39.90 81.00, 50.10 81.00, 50.10 87.00, 39.90 87.00, 39.90 81.00))')) AND reports < 20 AND active AND installation > TIMESTAMPADD(HOUR,NOW(),-9)";
	TrafficLight trafficLight;
	GSQuery* query;
	GSRowSet* rowSet;

	gsQuery(collection,tql,&query);
	gsFetch(query,GS_FALSE,&rowSet);

	// Create, issue, and fetch Query
	printf("\nPerforming 2D Intersection TQL Query: %s\n\n",tql);

	// Obtain and output its resulting rows
	while(gsHasNextRow(rowSet)){
		gsGetNextRow(rowSet,&trafficLight);
		GSChar timeStr[GS_TIME_STRING_SIZE_MAX];
		gsFormatTime(trafficLight.installation,timeStr,sizeof(timeStr));

		outputTrafficLight(trafficLight,timeStr);
	}

	gsDropCollection(gridstore,"Lights21");
	gsCloseGridStore(&gridstore, GS_TRUE);
}


int main(int argc, char** argv){
	if(argc < 6){
		printf("Need to enter GridDB host, port, cluster name, username, and password\n");
		return 1;
	} else {
		sample(argv[1],argv[2],argv[3],argv[4],argv[5]);
	}
	return 0;
}