package sample;

import java.lang.Object.*;
import java.io.IOException;
import java.io.FileReader;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import au.com.bytecode.opencsv.CSVReader;

import com.toshiba.mwcloud.gs.GridStore;
import com.toshiba.mwcloud.gs.GridStoreFactory;
import com.toshiba.mwcloud.gs.Geometry;
import com.toshiba.mwcloud.gs.GeometryOperator;
import com.toshiba.mwcloud.gs.TimestampUtils;
import com.toshiba.mwcloud.gs.TimeUnit;
import com.toshiba.mwcloud.gs.GSException;
import com.toshiba.mwcloud.gs.Collection;
import com.toshiba.mwcloud.gs.Query;
import com.toshiba.mwcloud.gs.IndexType;
import com.toshiba.mwcloud.gs.RowSet;

import sample.row.SmartBuilding;

import sample.logic.BoundingOption;
import sample.logic.Dimension;
import sample.logic.GeometryLogic;


public class BuildingApplication {
	private static final String COLLECTON_NAME = "SmartBuildings_101";
	private static final String COLUMN = "layout";
	private static final String DIRECTORY = "../data/";
	private static final String CSV_FILE = "Buildings.csv";

	private static double calculateVolume(double minX,double minY,double minZ,double maxX,double maxY,double maxZ){
		double xLength = Math.abs(maxX - minX);
		double yLength = Math.abs(maxY - minY);
		double zLength = Math.abs(maxZ - minZ);

		double volume = xLength * yLength * zLength;
		return volume;
	}

	private static void readCSV(Collection<String,SmartBuilding> collection, GeometryLogic geometryLogic, String filename) throws IOException, GSException {
		CSVReader reader = null;
		String nextLine[];

		try {
			reader = new CSVReader(new FileReader(DIRECTORY + filename));
			reader.readNext();

			while((nextLine = reader.readNext()) != null){
				SmartBuilding smartBuilding = new SmartBuilding();
				smartBuilding.buildingId = nextLine[0];

				double minX = Double.parseDouble(nextLine[1]);
				double minY = Double.parseDouble(nextLine[2]);
				double minZ = Double.parseDouble(nextLine[3]);
				double maxX = Double.parseDouble(nextLine[4]);
				double maxY = Double.parseDouble(nextLine[5]);
				double maxZ = Double.parseDouble(nextLine[6]);

				String surface = geometryLogic.formSurfaceWkt(minX,minY,minZ,maxX,maxY,maxZ);
				double volume = calculateVolume(minX,minY,minZ,maxX,maxY,maxZ);
				
				smartBuilding.layout = Geometry.valueOf(surface);
				smartBuilding.volume = volume;
				smartBuilding.installation = TimestampUtils.current();
				smartBuilding.name = nextLine[7];

				collection.put(smartBuilding.buildingId,smartBuilding);

				System.out.print("Inserted ");
				System.out.println(smartBuilding);
			}


		} catch(GSException e){
			e.printStackTrace();
			System.out.println("Error inserting records into GridDB");
		} catch(IOException e){
			e.printStackTrace();
			System.out.println("Error reading from " + filename);
		} finally {
			if(reader != null){
				reader.close();
			}
		}		
	}

	public static void main(String args[]) throws IOException, GSException{
		Properties properties = new Properties();
		properties.setProperty("notificationAddress",args[0]);
		properties.setProperty("notificationPort",args[1]);
		properties.setProperty("clusterName",args[2]);
		properties.setProperty("user",args[3]);
		properties.setProperty("password",args[4]);

		GridStore gridstore = null;
		gridstore = GridStoreFactory.getInstance().getGridStore(properties);

		Collection<String,SmartBuilding> buildingCollection = gridstore.putCollection(COLLECTON_NAME,SmartBuilding.class);
		System.out.println(String.format("Creating a spatial index on Column: %s",COLUMN));
		buildingCollection.createIndex(COLUMN,IndexType.SPATIAL);

		GeometryLogic geometryLogic = new GeometryLogic(buildingCollection,COLUMN);
		readCSV(buildingCollection,geometryLogic,CSV_FILE);


		System.out.println("\nSearching a sample volume");
		geometryLogic.searchVolume(0,5,0,5,0,5,true);

		System.out.println("\nSearching spatial ranges");
		geometryLogic.searchXRange(-5,-2);

		System.out.println("\nSearching a geographic area");
		geometryLogic.searchArea(10,0,15,12,true);

		System.out.println("\nSearching against a 3D plane.");
		List<Double> points = new ArrayList<>();

		points.add(0.0);
		points.add(0.0);
		points.add(0.0);
		points.add(1.0);
		points.add(1.0);
		points.add(0.0);
		points.add(2.0);
		points.add(1.5);
		points.add(0.5);
		points.add(1.0);
		points.add(1.0);
		points.add(1.0);

		geometryLogic.searchPlane(points,Dimension.THREE_DIMENSIONAL,true);

		System.out.println("\nTesting the against quadratic surfaces");
		double matrix[] = {0.5,0,0,0,0.5,0,0,0,0.5,0,0,0,-1,0,0,0};
		geometryLogic.searchQuadraticSurface(matrix,true);

		String wkt = "POLYHEDRALSURFACE(((3 4 -3, 7 8 -3, 10 11 -3, 3 4 -3)), ((7 8 -3, 10 11 -3, 2 2 -1, 7 8 -3)), ((3 4 -3, 7 8 -3, 2 2 -1, 3 4 -3)), ((3 4 -3, 10 11 -3, 2 2 -1, 3 4 -3)))";
		Geometry tetrahedron = Geometry.valueOf(wkt);

		Query<SmartBuilding> query = buildingCollection.query(COLUMN,tetrahedron,GeometryOperator.INTERSECT);
		RowSet<SmartBuilding> rowSet = query.fetch(false);

		System.out.println("\nChecking Geometry Query against a Tetrahedron");
		while(rowSet.hasNext()){
			SmartBuilding building = rowSet.next();
			System.out.println(building); 
		}


		gridstore.dropCollection(COLLECTON_NAME);
		if(gridstore != null){
			gridstore.close();
		}
	}
}
