CREATE EXTENSION griddb_fdw;
CREATE SERVER griddb_svr FOREIGN DATA WRAPPER griddb_fdw OPTIONS(host '239.0.0.1', port '31999', clustername 'defaultCluster');
CREATE USER MAPPING FOR public SERVER griddb_svr OPTIONS(username 'admin', password 'admin');

IMPORT FOREIGN SCHEMA griddb_schema FROM SERVER griddb_svr INTO public;
-- GridDB containers must be created for this test on GridDB server

CREATE FOREIGN TABLE department (department_id int, department_name text) SERVER griddb_svr;
CREATE FOREIGN TABLE employee (emp_id int, emp_name text, emp_dept_id int) SERVER griddb_svr;
CREATE FOREIGN TABLE empdata (emp_id int, emp_dat bytea) SERVER griddb_svr;
CREATE FOREIGN TABLE numbers (a int, b text) SERVER griddb_svr;
CREATE FOREIGN TABLE shorty (id int, c text) SERVER griddb_svr;

CREATE FOREIGN TABLE jp_prefecture (id text, name text, latitude float, longitude float, hascamera float) SERVER griddb_svr;
CREATE FOREIGN TABLE jp_prefecture_readings (timestamp timestamp, weatherstationid text, temperature float, hascamera boolean) SERVER griddb_svr;
CREATE FOREIGN TABLE water_quality_ts_3 (timestamp timestamp, water_temperature float, turbidity float, depth float, wave_height float, wave_period float, battery_life float, beach_name text) SERVER griddb_svr;

-- See all available containers from GridDB that can be used by PostgreSQL
SELECT foreign_table_name AS griddb_container FROM information_schema._pg_foreign_tables ORDER BY foreign_table_name ASC;

-- Verify Foreign Tables
SELECT * FROM jp_prefecture LIMIT 20;
SELECT timestamp, weatherstationid, temperature FROM jp_prefecture_readings LIMIT 100;
SELECT * FROM water_quality_ts_3 LIMIT 5;

-- Test Update Queries
UPDATE water_quality_ts_3 SET water_temperature = 18.5 WHERE beach_name LIKE '%met Beach';
UPDATE jp_prefecture SET latitude = 48.83, longitude = 140.75 WHERE id = 'weather_station_2' AND name='Aomori-Aomori';

-- Test Insertion Queries
INSERT INTO jp_prefecture (id,name,latitude,longitude,hascamera) 
  VALUES ('weather_station_48','Tsukishima',35.662,139.776,false);

INSERT INTO jp_prefecture_readings (timestamp, weatherstationid, temperature, liveimage) 
  VALUES('2018-05-20 12:35:55.789','weather_station_0',55.6,bytea('\x533435363741'));

DELETE FROM department;
DELETE FROM employee;
DELETE FROM empdata;

SELECT * FROM department LIMIT 10;
SELECT * FROM employee LIMIT 10;
SELECT * FROM empdata LIMIT 10;

INSERT INTO department VALUES(generate_series(1,100), 'dept - ' || generate_series(1,100));
INSERT INTO employee VALUES(generate_series(1,100), 'emp - ' || generate_series(1,100), generate_series(1,100));
INSERT INTO empdata  VALUES(1, bytea('\x0123456798765432'));

INSERT INTO numbers VALUES(0,'Zero');
INSERT INTO numbers VALUES(1, 'One');
INSERT INTO numbers VALUES(2, 'Two');
INSERT INTO numbers VALUES(3, 'Three');
INSERT INTO numbers VALUES(4, 'Four');
INSERT INTO numbers VALUES(5, 'Five');
INSERT INTO numbers VALUES(6, 'Six');
INSERT INTO numbers VALUES(7, 'Seven');
INSERT INTO numbers VALUES(8, 'Eight');
INSERT INTO numbers VALUES(9, 'Nine');
INSERT INTO numbers VALUES(10, 'Ten');
INSERT INTO numbers VALUES(11, 'Eleven');
INSERT INTO numbers VALUES(12, 'Twelve');
INSERT INTO numbers VALUES(13, 'Thirteen');
INSERT INTO numbers VALUES(14, 'Fourteen');
INSERT INTO numbers VALUES(15, 'Fifteen');
INSERT INTO numbers VALUES(16, 'Sixteen');
INSERT INTO numbers VALUES(17, 'Seventeen');
INSERT INTO numbers VALUES(18, 'Eighteen');
INSERT INTO numbers VALUES(19,'Nineteen');
INSERT INTO numbers VALUES(20,'Twenty');

SELECT count(*) FROM department;
SELECT count(*) FROM employee;
SELECT count(*) FROM empdata;

-- Join
SELECT * FROM department d, employee e WHERE d.department_id = e.emp_dept_id LIMIT 10;
-- Subquery
SELECT * FROM department d, employee e WHERE d.department_id IN (SELECT department_id FROM department) LIMIT 10;
SELECT * FROM empdata;
-- Delete single row
DELETE FROM employee WHERE emp_id = 10;

SELECT COUNT(*) FROM department LIMIT 10;
SELECT COUNT(*) FROM employee WHERE emp_id = 10;
-- Update single row
UPDATE employee SET emp_name = 'Updated emp' WHERE emp_id = 20;
SELECT emp_id, emp_name FROM employee WHERE emp_name like 'Updated emp';

UPDATE empdata SET emp_dat = bytea('\x4321');
SELECT * FROM empdata;

SELECT * FROM employee LIMIT 10;
SELECT * FROM employee WHERE emp_id IN (1);
SELECT * FROM employee WHERE emp_id IN (1,3,4,5);
SELECT * FROM employee WHERE emp_id IN (10000,1000);

SELECT * FROM employee WHERE emp_id NOT IN (1) LIMIT 5;
SELECT * FROM employee WHERE emp_id NOT IN (1,3,4,5) LIMIT 5;
SELECT * FROM employee WHERE emp_id NOT IN (10000,1000) LIMIT 5;

SELECT * FROM employee WHERE emp_id NOT IN (SELECT emp_id FROM employee WHERE emp_id IN (1,10));
SELECT * FROM employee WHERE emp_name NOT IN ('emp - 1', 'emp - 2') LIMIT 5;
SELECT * FROM employee WHERE emp_name NOT IN ('emp - 10') LIMIT 5;

CREATE OR REPLACE FUNCTION test_param_where() RETURNS void AS $$
DECLARE
  n varchar;
BEGIN
  FOR x IN 0..20 LOOP
    SELECT b INTO n FROM numbers WHERE a=x;
    RAISE NOTICE 'Found number %', n;
  END LOOP;
  RETURN;
END
$$ LANGUAGE plpgsql;

SELECT test_param_where();

DELETE FROM employee;
DELETE FROM department;
DELETE FROM empdata;
DELETE FROM numbers;

DROP FUNCTION test_param_where();
DROP FOREIGN TABLE numbers;
DROP FOREIGN TABLE department;
DROP FOREIGN TABLE employee;
DROP FOREIGN TABLE empdata;

-- -----------------------------------------------------------------------------
DELETE FROM shorty;
INSERT INTO shorty (id, c) VALUES (1, 'Z');
INSERT INTO shorty (id, c) VALUES (2, 'Y');
INSERT INTO shorty (id, c) VALUES (5, 'A');
INSERT INTO shorty (id, c) VALUES (3, 'X');
INSERT INTO shorty (id, c) VALUES (4, 'B');

-- ORDER BY.
SELECT c FROM shorty ORDER BY id;

-- Transaction INSERT
BEGIN;
INSERT INTO shorty (id, c) VALUES (6, 'T');
ROLLBACK;
SELECT id, c FROM shorty;

-- Transaction UPDATE single row
BEGIN;
UPDATE shorty SET c = 'd' WHERE id = 5;
ROLLBACK;
SELECT id, c FROM shorty;

-- Transaction UPDATE all
BEGIN;
UPDATE shorty SET c = 'd';
ROLLBACK;
SELECT id, c FROM shorty;

-- Transaction DELETE single row
BEGIN;
DELETE FROM shorty WHERE id = 1;
ROLLBACK;
SELECT id, c FROM shorty;

-- Transaction DELETE all
BEGIN;
DELETE FROM shorty;
ROLLBACK;
SELECT id, c FROM shorty;

-- parameters.
PREPARE stmt(integer) AS SELECT * FROM shorty WHERE id = $1;
EXECUTE stmt(1);
EXECUTE stmt(2);
DEALLOCATE stmt;

-- test NULL parameter
SELECT id FROM shorty WHERE c = (SELECT NULL::text);

-- Choosing Specific Columns FROM a Container
SELECT name, latitude, longitude FROM jp_prefecture;
SELECT timestamp, temperature FROM jp_prefecture_readings ORDER BY timestamp ASC;

-- Test Distinct SELECTION
SELECT DISTINCT weatherstationid FROM jp_prefecture_readings ORDER BY weatherstationid ASC;
SELECT DISTINCT temperature FROM jp_prefecture_readings ORDER BY temperature ASC;


-- Aggregation Operations coupled with Group By Operations
SELECT weatherstationid, AVG(temperature) AS average_temp FROM jp_prefecture_readings 
  GROUP BY weatherstationid ORDER BY average_temp ASC;

SELECT hascamera, COUNT(hascamera) FROM jp_prefecture GROUP BY hascamera; 

SELECT beach_name, AVG(water_temperature) AS avg_temp, MAX(wave_height), SUM(wave_period), MIN(battery_life)
  FROM water_quality_ts_3 GROUP BY beach_name ORDER BY avg_temp DESC;



-- Test Selecting FROM Multiple Table Selection
SELECT jp.id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature 
  FROM jp_prefecture jp, jp_prefecture_readings jpr WHERE jp.id = 'weather_station_18'
  ORDER BY jpr.timestamp DESC;

SELECT COUNT(*) FROM jp_prefecture, jp_prefecture_readings;

-- JOIN operation tests
SELECT jp.id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature
  FROM jp_prefecture jp, jp_prefecture_readings jpr WHERE jp.id = jpr.weatherstationid
  ORDER BY jpr.timestamp DESC;

SELECT jp.id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature
  FROM jp_prefecture jp INNER JOIN jp_prefecture_readings jpr ON jp.id = jpr.weatherstationid 
  ORDER BY jp.id ASC;

SELECT jp.id AS left_id, jpr.weatherstationid AS right_id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature
  FROM jp_prefecture jp LEFT OUTER JOIN jp_prefecture_readings jpr ON jp.id = jpr.weatherstationid
  ORDER BY jp.id ASC;

SELECT jp.id AS left_id, jpr.weatherstationid AS right_id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature
  FROM jp_prefecture jp RIGHT OUTER JOIN jp_prefecture_readings jpr ON jp.id = jpr.weatherstationid
  ORDER BY jp.id ASC;

SELECT jp.id AS left_id, jpr.weatherstationid AS right_id, jp.latitude, jp.longitude, jpr.timestamp, jpr.temperature
  FROM jp_prefecture jp FULL OUTER JOIN jp_prefecture_readings jpr ON jp.id = jpr.weatherstationid
  ORDER BY jp.id ASC;


-- Union Tests
SELECT id FROM jp_prefecture 
  UNION 
SELECT weatherstationid FROM jp_prefecture_readings;

SELECT id FROM jp_prefecture 
  UNION ALL
SELECT weatherstationid FROM jp_prefecture_readings;




DELETE FROM jp_prefecture WHERE name='Tsukishima';
DELETE FROM jp_prefecture_readings 
  WHERE weatherstationid LIKE '%0' AND temperature > 50.0 AND temperature <= 60.0;


-- Clean up
DROP FOREIGN TABLE shorty;

-- Release all Data from GridDB Containers
DELETE FROM water_quality_ts_3;
DELETE FROM jp_prefecture_readings;
DELETE FROM jp_prefecture;

DROP FOREIGN TABLE water_quality_ts_3;
DROP FOREIGN TABLE jp_prefecture_readings;
DROP FOREIGN TABLE jp_prefecture;


-- Clean Up Boilerplate Code from pgspider
CREATE OR REPLACE FUNCTION drop_all_foreign_tables() RETURNS void AS $$
DECLARE
  tbl_name varchar;
  cmd varchar;
BEGIN
  FOR tbl_name IN SELECT foreign_table_name FROM information_schema._pg_foreign_tables LOOP
    cmd := 'DROP FOREIGN TABLE ' || tbl_name;
    EXECUTE cmd;
  END LOOP;
  RETURN;
END
$$ LANGUAGE plpgsql;
SELECT drop_all_foreign_tables();

DROP USER MAPPING FOR public SERVER griddb_svr;
DROP SERVER griddb_svr CASCADE;
DROP EXTENSION griddb_fdw CASCADE;
