package sample;

import java.util.Properties;
import java.util.Date;
import java.util.Map;
import java.util.Map.Entry;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.EnumSet;

import java.lang.*;

import java.io.IOException;
import java.sql.SQLException;

import com.toshiba.mwcloud.gs.GridStore;
import com.toshiba.mwcloud.gs.GridStoreFactory;
import com.toshiba.mwcloud.gs.GSException;
import com.toshiba.mwcloud.gs.TimeSeries;
import com.toshiba.mwcloud.gs.ColumnInfo;
import com.toshiba.mwcloud.gs.ContainerInfo;
import com.toshiba.mwcloud.gs.IndexType;
import com.toshiba.mwcloud.gs.ContainerType;
import com.toshiba.mwcloud.gs.GSType;
import com.toshiba.mwcloud.gs.Collection;
import com.toshiba.mwcloud.gs.Container;
import com.toshiba.mwcloud.gs.TimestampUtils;
import com.toshiba.mwcloud.gs.Row;

import sample.row.Department;
import sample.row.Employee;
import sample.row.EmployeeData;
import sample.row.Number;
import sample.row.Shorty;

import sample.row.WeatherStation;
import sample.row.InstrumentLog;
import sample.row.QualitySample;

import sample.logic.WeatherStationLoader;
import sample.logic.InstrumentLogLoader;
import sample.logic.QualitySampleLoader;


public class LoadDatabase {
	private static final String PREFIX = "../data/";

	//Containers that will be used for test SQL queries for the GridDB Foreign Data Wrapper for PostgreSQL
	private static final String DEPARTMENT_TABLE  = "department";
	private static final String EMPLOYEE_TABLE = "employee";
	private static final String EMPLOYEE_DATA_TABLE = "empdata";
	private static final String NUMBER_TABLE = "numbers";
	private static final String SHORTY_TABLE = "shorty";

	private static final String keys[] = {"weather_station","instrument_log","quality_sample"};

	//Name of prepopulated GridDB Containers that will also be accessed by PostgreSQL through the Foreign Data Wrapper
	private static final String containers[] = {"jp_prefecture","jp_prefecture_readings","water_quality_ts_3"};

	//Name of the respective data source *.csv files for each GridDB Container
	private static final String sources[] = {"weather_station.csv","instrument_log.csv","water_quality_samples.csv"};

	private static Map<String,Map<String,String>> containerMappings;

	private static Map<String,GSType> columnTestSchemas; 


	private static void loadMappings(){
		int count = containers.length;
		if(count != sources.length || count != keys.length){
			return;
		}

		for(int i = 0; i < count; i++){
			Map<String,String> datum = new HashMap<>();
			datum.put("name",containers[i]);
			datum.put("source",PREFIX + sources[i]);

			containerMappings.put(keys[i],datum);
		}
	}

	private static void loadTestColumnSchemaMap(){
		//Foreign Tables / Containers that will be used to test Data/Column Types 
		//for the GridDB Foreign Data Wrapper for PostgreSQL
		//All have integer (primary keys / row keys)
		columnTestSchemas.put("type_string",GSType.STRING); //Test string columns
		columnTestSchemas.put("type_boolean",GSType.BOOL); //Test boolean columns
		columnTestSchemas.put("type_byte",GSType.BYTE); //Test byte columns
		columnTestSchemas.put("type_short",GSType.SHORT); //Test short columns
		columnTestSchemas.put("type_integer",GSType.INTEGER); //Test integer columns
		columnTestSchemas.put("type_long",GSType.LONG); //Test long columns
		columnTestSchemas.put("type_float",GSType.FLOAT); //Test floating point columns
		columnTestSchemas.put("type_double",GSType.DOUBLE); //Test double precision columns
		columnTestSchemas.put("type_timestamp",GSType.TIMESTAMP); //Test timestamp columns
		columnTestSchemas.put("type_blob",GSType.BLOB); //Test Blob / Binary stream data columns
		columnTestSchemas.put("type_string_array",GSType.STRING_ARRAY); //Test string array columns
		columnTestSchemas.put("type_bool_array",GSType.BOOL_ARRAY); //Test boolean array columns
		columnTestSchemas.put("type_byte_array",GSType.BYTE_ARRAY); //Test byte array columns
		columnTestSchemas.put("type_short_array",GSType.SHORT_ARRAY); //Test short array columns
		columnTestSchemas.put("type_integer_array",GSType.INTEGER_ARRAY); //Test integer array columns
		columnTestSchemas.put("type_long_array",GSType.LONG_ARRAY); //Test long array columns
		columnTestSchemas.put("type_float_array",GSType.FLOAT_ARRAY); //Test floating point array columns
		columnTestSchemas.put("type_double_array",GSType.DOUBLE_ARRAY); //Test double precision array columns
		columnTestSchemas.put("type_timestamp_array",GSType.TIMESTAMP_ARRAY); //Test timestamp array columns
	}

	private static void loadTestContainers(GridStore gridstore) throws GSException {
		if(gridstore != null){
			for(Entry<String,GSType> entry : columnTestSchemas.entrySet()){
				String containerName = entry.getKey(); //Get name of test container
				GSType type = entry.getValue();  //Get the GridDB Column Type that it will use for testing in the *.sql scripts in the 'sql' directory

				List<ColumnInfo> columnInfoList = new ArrayList<ColumnInfo>(); //Column Types for the container
				EnumSet<IndexType> indexSet = EnumSet.of(IndexType.TREE); //Create an index and an 'integer' primary key / row key.
				ColumnInfo keyColumn = new ColumnInfo("id",GSType.INTEGER,indexSet); //Set row key column

				columnInfoList.add(keyColumn); //Assign a column named 'col' that will test for a certain column type in GridDB 
												//against the GridDB Foreign Data Wrapper for PostgreSQL
				columnInfoList.add(new ColumnInfo("col",type)); //Set column

				ContainerInfo containerInfo = new ContainerInfo(containerName,ContainerType.COLLECTION,columnInfoList,true); //Create a container schema
				Container<Integer,Row> container = gridstore.putContainer(containerName,containerInfo,true); //Create a container in GridDB for the with the above schema

				System.out.println("Successfully entered test container " + containerName + " into GridDB");
			}
		} else {
			System.out.println("Entered an invalid instance of GridDB");
		}
	}

	private static void load(GridStore gridstore) throws GSException {
		if(gridstore != null){
			//Load containers into GridDB that will be used by PostgreSQL through the foreign data wrapper
			Container<Integer,Department> departmentContainer = gridstore.putCollection(DEPARTMENT_TABLE,Department.class); //GridDB Container for foreign table 'department' used by PostgreSQL
			System.out.println("Container: " + DEPARTMENT_TABLE + " has been created");

			Container<Integer,Employee> employeeContainer = gridstore.putCollection(EMPLOYEE_TABLE,Employee.class); //GridDB Container for foreign table 'employee' used by PostgreSQL
			System.out.println("Container: " + EMPLOYEE_TABLE + " has been created");

			Container<Integer,EmployeeData> employeeDataContainer = gridstore.putCollection(EMPLOYEE_DATA_TABLE,EmployeeData.class); //GridDB Container for foreign table 'emp_data' used by PostgreSQL
			System.out.println("Container: " + EMPLOYEE_DATA_TABLE + " has been created");

			Container<Integer,Number> numberContainer = gridstore.putCollection(NUMBER_TABLE,Number.class); //GridDB Container for foreign table 'number' used by PostgreSQL
			System.out.println("Container: " + NUMBER_TABLE + " has been created");

			Container<Integer,Shorty> shortyContainer = gridstore.putCollection(SHORTY_TABLE,Shorty.class); //GridDB Container for foreign table 'shorty' used by PostgreSQL
			System.out.println("Container: " + SHORTY_TABLE + " has been created");

			System.out.println("All tables have been successfully created!");
		} else {
			System.out.println("Entered an invalid instance of GridDB");
		}
	}

	public static void main(String args[]) throws GSException, SQLException, IOException {
		Properties properties = new Properties();

		//Collect GridDB Cluster configuration settings from the command line
		properties.setProperty("notificationAddress",args[0]);
		properties.setProperty("notificationPort",args[1]);
		properties.setProperty("clusterName",args[2]);
		properties.setProperty("user",args[3]);
		properties.setProperty("password",args[4]);

		GridStore gridstore = null;		
		
		containerMappings = new HashMap<>();
		columnTestSchemas = new HashMap<>();

		try {
			loadMappings();
			loadTestColumnSchemaMap();

			gridstore = GridStoreFactory.getInstance().getGridStore(properties); //Connect to GridDB

			//Load containers that will be accessed as Foreign Tables in PostgreSQL through the GridDB's PostgreSQL Foreign Data Wrapper into GridDB 
			load(gridstore); 
			loadTestContainers(gridstore);


			for(Entry<String,Map<String,String>> entry : containerMappings.entrySet()){
				System.out.println("Key: " + entry.getKey());

				for(Entry<String,String> e : entry.getValue().entrySet()){
					System.out.println("\t" + e.getKey() + " : " + e.getValue());
				}
			}

			//Load Weather Station Wrapper into the Weather Station Collection Container
			WeatherStationLoader.load(gridstore,containerMappings.get(keys[0]).get("name"),containerMappings.get(keys[0]).get("source"));
			System.out.println("Loaded weather station collection container");

			//Load Weather Station temperature readings (timestamped) into Instrument Log Timeseries Container
			InstrumentLogLoader.load(gridstore,containerMappings.get(keys[1]).get("name"),containerMappings.get(keys[1]).get("source"));
			System.out.println("Loaded instrument log timeseries container");

			//Load timestamped water quality measurements Water Quality Timeseries container.
			QualitySampleLoader.load(gridstore,containerMappings.get(keys[2]).get("name"),containerMappings.get(keys[2]).get("source"));
			System.out.println("Loaded Water Quality Sample timeseries container");



		} catch(Exception e){
			e.printStackTrace();
		} finally {
			if(gridstore != null){
				gridstore.close(); //Close access to GridDB
			}
		}

	}

}