/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.jcraft.eclipse.jsch.core;

import java.io.PrintStream;
import java.lang.reflect.Field;

import org.eclipse.core.runtime.*;


public class Policy{
  public static PrintStream recorder;

  //debug constants
  public static boolean DEBUG_METAFILE_CHANGES=false;
  public static boolean DEBUG_CVS_PROTOCOL=false;
  public static boolean DEBUG_THREADING=false;
  public static boolean DEBUG_DIRTY_CACHING=false;
  public static boolean DEBUG_SYNC_CHANGE_EVENTS=false;

  static{
    //init debug options
    if(JSchCorePlugin.getPlugin().isDebugging()){
      DEBUG_METAFILE_CHANGES="true".equalsIgnoreCase(Platform.getDebugOption(JSchCorePlugin.ID+"/metafiles"));//$NON-NLS-1$ //$NON-NLS-2$
      DEBUG_CVS_PROTOCOL="true".equalsIgnoreCase(Platform.getDebugOption(JSchCorePlugin.ID+"/cvsprotocol"));//$NON-NLS-1$ //$NON-NLS-2$
      DEBUG_THREADING="true".equalsIgnoreCase(Platform.getDebugOption(JSchCorePlugin.ID+"/threading"));//$NON-NLS-1$ //$NON-NLS-2$
      DEBUG_DIRTY_CACHING="true".equalsIgnoreCase(Platform.getDebugOption(JSchCorePlugin.ID+"/dirtycaching"));//$NON-NLS-1$ //$NON-NLS-2$
      DEBUG_SYNC_CHANGE_EVENTS="true".equalsIgnoreCase(Platform.getDebugOption(JSchCorePlugin.ID+"/syncchangeevents"));//$NON-NLS-1$ //$NON-NLS-2$
    }
  }

  /**
   * Progress monitor helpers
   */
  public static void checkCanceled(IProgressMonitor monitor){
    if(monitor.isCanceled())
      throw new OperationCanceledException();
  }

  public static IProgressMonitor monitorFor(IProgressMonitor monitor){
    if(monitor==null)
      return new NullProgressMonitor();
    return monitor;
  }

  public static IProgressMonitor subMonitorFor(IProgressMonitor monitor,
      int ticks){
    if(monitor==null)
      return new NullProgressMonitor();
    if(monitor instanceof NullProgressMonitor)
      return monitor;
    return new SubProgressMonitor(monitor, ticks);
  }

  public static IProgressMonitor infiniteSubMonitorFor(
      IProgressMonitor monitor, int ticks){
    if(monitor==null)
      return new NullProgressMonitor();
    if(monitor instanceof NullProgressMonitor)
      return monitor;
    return new InfiniteSubProgressMonitor(monitor, ticks);
  }

  public static boolean isDebugProtocol(){
    return DEBUG_CVS_PROTOCOL||recorder!=null;
  }

  public static void printProtocolLine(String line){
    printProtocol(line, true);
  }

  public static void printProtocol(String string, boolean newLine){
    if(DEBUG_CVS_PROTOCOL){
      System.out.print(string);
      if(newLine){
        System.out.println();
      }
    }
    if(recorder!=null){
      recorder.print(string);
      if(newLine){
        recorder.println();
      }
    }
  }

  public static String getMessage(String key){
    try{
      Field f=Messages.class.getDeclaredField(key);
      Object o=f.get(null);
      if(o instanceof String)
        return (String)o;
    }
    catch(SecurityException e){
    }
    catch(NoSuchFieldException e){
    }
    catch(IllegalArgumentException e){
    }
    catch(IllegalAccessException e){
    }
    return null;
  }
  
  /**
   * Provides an infinite progress monitor by subdividing by half repeatedly.
   * 
   * The ticks parameter represents the number of ticks shown in the progress dialog
   * (or propogated up to a parent IProgressMonitor). The totalWork parameter provided
   * in actually a hint used to determine how work is translated into ticks.
   * The number of totalWork that can actually be worked is n*totalWork/2 where
   * 2^n = totalWork. What this means is that if you provide a totalWork of 32 (2^5) than
   * the maximum number of ticks is 5*32/2 = 80.
   * 
   */
  static class InfiniteSubProgressMonitor extends SubProgressMonitor{

    int totalWork;
    int halfWay;
    int currentIncrement;
    int nextProgress;
    int worked;

    /**
     * Constructor for InfiniteSubProgressMonitor.
     * @param monitor
     * @param ticks
     */
    public InfiniteSubProgressMonitor(IProgressMonitor monitor, int ticks){
      this(monitor, ticks, 0);
    }

    /**
     * Constructor for InfiniteSubProgressMonitor.
     * @param monitor
     * @param ticks
     * @param style
     */
    public InfiniteSubProgressMonitor(IProgressMonitor monitor, int ticks,
        int style){
      super(monitor, ticks, style);
    }

    public void beginTask(String name, int totalWork){
      super.beginTask(name, totalWork);
      this.totalWork=totalWork;
      this.halfWay=totalWork/2;
      this.currentIncrement=1;
      this.nextProgress=currentIncrement;
      this.worked=0;
    }

    public void worked(int work){
      if(worked>=totalWork)
        return;
      if(--nextProgress<=0){
        super.worked(1);
        worked++;
        if(worked>=halfWay){
          // we have passed the current halfway point, so double the
          // increment and reset the halfway point.
          currentIncrement*=2;
          halfWay+=(totalWork-halfWay)/2;
        }
        // reset the progress counter to another full increment
        nextProgress=currentIncrement;
      }
    }

    /**
     * Don't allow clearing of the subtask. This will stop the flickering
     * of the subtask in the progress dialogs.
     * 
     * @see IProgressMonitor#subTask(String)
     */
    public void subTask(String name){
      if(name!=null&&!name.equals("")){ //$NON-NLS-1$
        super.subTask(name);
      }
    }
  }
}
