/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.jcraft.eclipse.jsch.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.*;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.INodeChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.NodeChangeEvent;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.PreferenceChangeEvent;
import org.osgi.service.prefs.BackingStoreException;


/**
 * This class keeps track of the CVS repository locations that are known to
 * the CVS plugin.
 */
public class KnownRepositories implements INodeChangeListener, IPreferenceChangeListener {

	private List repositoryListeners = new ArrayList();
	private Map repositories;

	private static KnownRepositories instance;
	
	public static synchronized KnownRepositories getInstance() {
		if (instance == null) {
			instance = new KnownRepositories();
		}
		return instance;
	}
	
	/*
	 * Private class used to safely notify listeners of resouce sync info changes. 
	 * Subclass override the notify(IResourceStateChangeListener) method to
	 * fire specific events inside an ISafeRunnable.
	 */
	private abstract class Notification implements ISafeRunnable {
		private IJSchCoreListener listener;
		public void handleException(Throwable exception) {
			// don't log the exception....it is already being logged in Platform#run
		}
		public void run(IJSchCoreListener listener) {
			this.listener = listener;
      SafeRunner.run(this);
		}
		public void run() throws Exception {
			notify(listener);
		}
		/**
		 * Subsclasses overide this method to send an event safely to a lsistener
		 * @param listener
		 */
		protected abstract void notify(IJSchCoreListener listener);
	}
	
	/**
	 * Register to receive notification of repository creation and disposal
	 */
	public void addRepositoryListener(IJSchCoreListener listener) {
		synchronized(repositoryListeners) {
			repositoryListeners.add(listener);
		}
	}
	
	/**
	 * De-register a listener
	 */
	public void removeRepositoryListener(IJSchCoreListener listener) {
		synchronized(repositoryListeners) {
			repositoryListeners.remove(listener);
		}
	}

	/**
	 * Add the repository to the receiver's list of known repositories. Doing this will enable
	 * password caching across platform invocations.
	 */
	public IJSchLocation addRepository(final IJSchLocation repository, boolean broadcast) {
		JSchLocation existingLocation;
		synchronized (this) {
			// Check the cache for an equivalent instance and if there is one, just update the cache
			existingLocation = internalGetRepository(repository.getLocation(false));
			if (existingLocation == null) {
				// Store the location
				store((JSchLocation)repository);
				existingLocation = (JSchLocation)repository;
			}
		}
		// Notify no matter what since it may not have been broadcast before
		if (broadcast) {
			final JSchLocation location = existingLocation;
			((JSchLocation)repository).updateCache();
			fireNotification(new Notification() {
				public void notify(IJSchCoreListener listener) {
					listener.repositoryAdded(location);
				}
			});
		}
		return existingLocation;
	}
	
	/**
	 * Dispose of the repository location
	 * 
	 * Removes any cached information about the repository such as a remembered password.
	 */
	public void disposeRepository(final IJSchLocation repository) {
		Object removed;
		synchronized (this) {
			((JSchLocation)repository).dispose();
			removed = getRepositoriesMap().remove(repository.getLocation(false));
		}
		if (removed != null) {
			fireNotification(new Notification() {
				public void notify(IJSchCoreListener listener) {
					listener.repositoryRemoved(repository);
				}
			});
		}
	}

	/**
	 * Answer whether the provided repository location is known by the provider or not.
	 * The location string corresponds to the String returned by ICVSRepositoryLocation#getLocation()
	 */
	public synchronized boolean isKnownRepository(String location) {
		return internalGetRepository(location) != null;
	}

	/** 
	 * Return a list of the know repository locations
	 */
	public synchronized IJSchLocation[] getRepositories() {
		return (IJSchLocation[])getRepositoriesMap().values().toArray(new IJSchLocation[getRepositoriesMap().size()]);
	}
	
	/**
	 * Get the repository instance which matches the given String. The format of the String is
	 * the same as that returned by ICVSRepositoryLocation#getLocation().
	 * The format is:
	 * 
	 *   connection:user[:password]@host[#port]:root
	 * 
	 * where [] indicates optional and the identier meanings are:
	 * 
	 * 	 connection The connection method to be used
	 *   user The username for the connection
	 *   password The password used for the connection (optional)
	 *   host The host where the repository resides
	 *   port The port to connect to (optional)
	 *   root The server directory where the repository is located
	 * 
	 * If the repository is already registered, the cahced instance is returned.
	 * Otherwise, a new uncached instance is returned.
	 * 
	 * WARNING: Providing the password as part of the String will result in the password being part
	 * of the location permanently. This means that it cannot be modified by the authenticator. 
	 */
  
	public synchronized IJSchLocation getRepository(String location) throws JSchCoreException {
		IJSchLocation repository = internalGetRepository(location);
		if (repository == null) {
			repository = JSchLocation.fromString(location);
		}
		return repository;
	}
	
	private JSchLocation internalGetRepository(String location) {
		return (JSchLocation)getRepositoriesMap().get(location);
	}
	
	/*
	 * Cache the location and store it in the preferences for persistance
	 */
	private void store(JSchLocation location) {
		// Cache the location instance for later retrieval
		getRepositoriesMap().put(location.getLocation(), location);
		location.storePreferences();
	}

	private Map getRepositoriesMap() {
		if (repositories == null) {
			// Load the repositories from the preferences
			repositories = new HashMap();
			IEclipsePreferences prefs = (IEclipsePreferences) JSchLocation.getParentPreferences();
			prefs.addNodeChangeListener(this);
			try {
				String[] keys = prefs.childrenNames();
				for (int i = 0; i < keys.length; i++) {
					String key = keys[i];
					try {
						IEclipsePreferences node = (IEclipsePreferences) prefs.node(key);
						node.addPreferenceChangeListener(this);
						String location = node.get(JSchLocation.PREF_LOCATION, null);
						if (location != null) {
							repositories.put(location, JSchLocation.fromString(location));
						} else {
							node.removeNode();
							prefs.flush();
						}
					} catch (JSchCoreException e) {
						// Log and continue
						JSchCorePlugin.log(e);
					}
				}
				if (repositories.isEmpty()) {
          // TODO
//					getRepositoriesFromProjects();
				}
			} catch (BackingStoreException e) {
				// Log and continue (although all repos will be missing)
				JSchCorePlugin.log(IStatus.ERROR, Messages.KnownRepositories_0, e); 
			} catch (Exception e) {
				//JSchCorePlugin.log(e);
			}
		}
		return repositories;
	}
	
  
  /*
	private void getRepositoriesFromProjects() throws CVSException {
		// If the file did not exist, then prime the list of repositories with
		// the providers with which the projects in the workspace are shared.
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		for (int i = 0; i < projects.length; i++) {
			RepositoryProvider provider = RepositoryProvider.getProvider(projects[i], JSchCorePlugin.getTypeId());
			if (provider!=null) {
				ICVSFolder folder = (ICVSFolder)CVSWorkspaceRoot.getCVSResourceFor(projects[i]);
				FolderSyncInfo info = folder.getFolderSyncInfo();
				if (info != null) {
					addRepository(getRepository(info.getRoot()), false);
				}
			}
		}
	}
	*/
  
	private IJSchCoreListener[] getListeners() {
		synchronized(repositoryListeners) {
			return (IJSchCoreListener[]) repositoryListeners.toArray(new IJSchCoreListener[repositoryListeners.size()]);
		}
	}
	
	private void fireNotification(Notification notification) {
		// Get a snapshot of the listeners so the list doesn't change while we're firing
		IJSchCoreListener[] listeners = getListeners();
		// Notify each listener in a safe manner (i.e. so their exceptions don't kill us)
		for (int i = 0; i < listeners.length; i++) {
			IJSchCoreListener listener = listeners[i];
			notification.run(listener);
		}
	}

	public void added(NodeChangeEvent event) {
		((IEclipsePreferences)event.getChild()).addPreferenceChangeListener(this);
	}

	public void removed(NodeChangeEvent event) {
		// Cannot remove the listener once the node is removed
		//((IEclipsePreferences)event.getChild()).removePreferenceChangeListener(this);
	}

	public void preferenceChange(PreferenceChangeEvent event) {
		if (JSchLocation.PREF_LOCATION.equals(event.getKey())) {
			String location = (String)event.getNewValue();
			if (location == null) {
				((IEclipsePreferences)event.getNode()).removePreferenceChangeListener(this);
			} else {
				try {
					addRepository(JSchLocation.fromString(location), true);
				} catch (JSchCoreException e) {
					JSchCorePlugin.log(e);
				}
			}
		}
	}
}
