/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Atsuhiko Yamanaka, JCraft,Inc. - initial API and implementation.
 *     IBM Corporation - ongoing maintenance
 *******************************************************************************/
package com.jcraft.eclipse.jsch.core;

import java.io.*;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.core.runtime.*;

import org.eclipse.jsch.core.IJSchService;
import org.eclipse.osgi.util.NLS;

import com.jcraft.jsch.*;

public class JSchSession{
  private static final int SSH_DEFAULT_PORT=22;
  private static JSchContext defaultContext=null;
  static{
    defaultContext=new JSchContext();
  }
  public static JSchContext getDefaultContext(){
    return defaultContext;
  }

  private final Session session;
  private final com.jcraft.jsch.UserInfo prompter;
  private final IJSchLocation location;

  protected static int getCVSTimeoutInMillis(){
    //return CVSProviderPlugin.getPlugin().getTimeout() * 1000;
    // TODO Hard-code the timeout for now since Jsch doesn't respect CVS timeout
    // See bug 92887
    return 60000;
  }

  public static class SimpleSocketFactory implements SocketFactory{
    InputStream in=null;
    OutputStream out=null;

    public Socket createSocket(String host, int port) throws IOException,
        UnknownHostException{
      Socket socket=null;
      socket=new Socket(host, port);
      return socket;
    }

    public InputStream getInputStream(Socket socket) throws IOException{
      if(in==null)
        in=socket.getInputStream();
      return in;
    }

    public OutputStream getOutputStream(Socket socket) throws IOException{
      if(out==null)
        out=socket.getOutputStream();
      return out;
    }
  }

  public static class ResponsiveSocketFacory extends SimpleSocketFactory{
    private IProgressMonitor monitor;

    public ResponsiveSocketFacory(IProgressMonitor monitor){
      this.monitor=monitor;
    }

    public Socket createSocket(String host, int port) throws IOException,
        UnknownHostException{
      Socket socket=null;
      //socket=Util.createSocket(host, port, monitor);
      socket=createSocketTimeout(host, port, monitor);
      // Null out the monitor so we don't hold onto anything
      // (i.e. the SSH2 session will keep a handle to the socket factory around
      monitor=new NullProgressMonitor();
      // Set the socket timeout
      socket.setSoTimeout(getCVSTimeoutInMillis());
      return socket;
    }
  }

  /**
   * Helper method that will time out when making a socket connection.
   * This is required because there is no way to provide a timeout value
   * when creating a socket and in some instances, they don't seem to
   * timeout at all.
   */
  private static Socket createSocketTimeout(final String host, final int port,
      IProgressMonitor monitor) throws UnknownHostException, IOException{

    // Start a thread to open a socket
    final Socket[] socket=new Socket[] {null};
    final Exception[] exception=new Exception[] {null};
    final Thread thread=new Thread(new Runnable(){
      public void run(){
        try{
          Socket newSocket=new Socket(host, port);
          synchronized(socket){
            if(Thread.interrupted()){
              // we we're either cancelled or timed out so just close the socket
              newSocket.close();
            }
            else{
              socket[0]=newSocket;
            }
          }
        }
        catch(UnknownHostException e){
          exception[0]=e;
        }
        catch(IOException e){
          exception[0]=e;
        }
      }
    });
    thread.start();

    // Wait the appropriate number of seconds
    // TODO
    int timeout = JSchCorePlugin.getPlugin().getTimeout();
    if (timeout == 0) timeout = JSchCorePlugin.DEFAULT_TIMEOUT;
    //int timeout=1000;
    //if(timeout==0)
      //timeout=1000;

    for(int i=0; i<timeout; i++){
      try{
        // wait for the thread to complete or 1 second, which ever comes first
        thread.join(1000);
      }
      catch(InterruptedException e){
        // I think this means the thread was interupted but not necessarily timed out
        // so we don't need to do anything
      }
      synchronized(socket){
        // if the user cancelled, clean up before preempting the operation
        if(monitor.isCanceled()){
          if(thread.isAlive()){
            thread.interrupt();
          }
          if(socket[0]!=null){
            socket[0].close();
          }
          // this method will throw the proper exception
          Policy.checkCanceled(monitor);
        }
      }
    }
    // If the thread is still running (i.e. we timed out) signal that it is too late
    synchronized(socket){
      if(thread.isAlive()){
        thread.interrupt();
      }
    }
    if(exception[0]!=null){
      if(exception[0] instanceof UnknownHostException)
        throw (UnknownHostException)exception[0];
      else
        throw (IOException)exception[0];
    }
    if(socket[0]==null){
      throw new InterruptedIOException(NLS.bind(Messages.Util_timeout,
          new String[] {host}));
    }
    return socket[0];
  }

  /**
   * UserInfo wrapper class that will time how long each prompt takes
   */
  private static class UserInfoTimer implements com.jcraft.jsch.UserInfo, UIKeyboardInteractive{

    private com.jcraft.jsch.UserInfo wrappedInfo;
    private long startTime;
    private long endTime;
    private boolean prompting;

    public UserInfoTimer(com.jcraft.jsch.UserInfo wrappedInfo){
      this.wrappedInfo=wrappedInfo;
    }

    private synchronized void startTimer(){
      prompting=true;
      startTime=System.currentTimeMillis();
    }

    private synchronized void endTimer(){
      prompting=false;
      endTime=System.currentTimeMillis();
    }

    public long getLastDuration(){
      return Math.max(0, endTime-startTime);
    }

    public boolean hasPromptExceededTimeout(){
      if(!isPrompting()){
        return getLastDuration()>getCVSTimeoutInMillis();
      }
      return false;
    }

    public String getPassphrase(){
      return wrappedInfo.getPassphrase();
    }

    public String getPassword(){
      return wrappedInfo.getPassword();
    }

    public boolean promptPassword(String arg0){
      try{
        startTimer();
        return wrappedInfo.promptPassword(arg0);
      }
      finally{
        endTimer();
      }
    }

    public boolean promptPassphrase(String arg0){
      try{
        startTimer();
        return wrappedInfo.promptPassphrase(arg0);
      }
      finally{
        endTimer();
      }
    }

    public boolean promptYesNo(String arg0){
      try{
        startTimer();
        return wrappedInfo.promptYesNo(arg0);
      }
      finally{
        endTimer();
      }
    }

    public void showMessage(String arg0){
      if(arg0.length()!=0){
        try{
          startTimer();
          wrappedInfo.showMessage(arg0);
        }
        finally{
          endTimer();
        }
      }
    }

    public String[] promptKeyboardInteractive(String arg0, String arg1,
        String arg2, String[] arg3, boolean[] arg4){
      try{
        startTimer();
        return ((UIKeyboardInteractive)wrappedInfo).promptKeyboardInteractive(
            arg0, arg1, arg2, arg3, arg4);
      }
      finally{
        endTimer();
      }
    }

    public boolean isPrompting(){
      return prompting;
    }
  }

  /**
   * User information delegates to the IUserAuthenticator. This allows
   * headless access to the connection method.
   */
  private static class MyUserInfo implements com.jcraft.jsch.UserInfo, UIKeyboardInteractive{
    private String username;
    private String password;
    private String passphrase;
    private IJSchLocation location;
    private IUserAuthenticator authenticator;
    private int attemptCount;
    private boolean passwordChanged;

    MyUserInfo(String username, String password, IJSchLocation location, IUserAuthenticator authenticator){
      this.location=location;
      this.username=username;
      this.password=password;
      this.authenticator=authenticator;
      if(this.authenticator==null)
        this.authenticator=JSchCorePlugin.getPluggedInAuthenticator();
    }

    public String getPassword(){
      return password;
    }

    public String getPassphrase(){
      return passphrase;
    }

    public boolean promptYesNo(String str){
      int prompt=authenticator.prompt(location, IUserAuthenticator.QUESTION,
          Messages.JSchSession_5, str, new int[] {IUserAuthenticator.YES_ID,
              IUserAuthenticator.NO_ID}, 0 //yes the default
          );
      return prompt==0;
    }

    private String promptSecret(String message, boolean includeLocation)
        throws JSchCoreException{
      final String[] _password=new String[1];
      IUserInfo info=new IUserInfo(){
        public String getUsername(){
          return username;
        }

        public boolean isUsernameMutable(){
          return false;
        }

        public void setPassword(String password){
          _password[0]=password;
        }

        public void setUsername(String username){
        }
      };
      try{
        authenticator.promptForUserInfo(includeLocation ? location : null,
            info, message);
      }
      catch(OperationCanceledException e){
        _password[0]=null;
      }
      return _password[0];
    }

    public boolean promptPassphrase(String message){
      try{
        String _passphrase=promptSecret(message, false);
        if(_passphrase!=null){
          passphrase=_passphrase;
        }
        return _passphrase!=null;
      }
      catch(JSchCoreException e){
        return false;
      }
    }

    public boolean promptPassword(String message){
      try{
        String _password=promptSecret(message, true);
        if(_password!=null){
          password=_password;
          // Cache the password with the repository location on the memory.
          if(location!=null)
            ((IJSchLocation)location).setPassword(password);
        }
        return _password!=null;
      }
      catch(JSchCoreException e){
        return false;
      }
    }

    public void showMessage(String message){
      authenticator.prompt(location, IUserAuthenticator.INFORMATION,
          Messages.JSchSession_5, message,
          new int[] {IUserAuthenticator.OK_ID}, IUserAuthenticator.OK_ID);

    }

    public String[] promptKeyboardInteractive(String destination, String name,
        String instruction, String[] prompt, boolean[] echo){
      if(prompt.length==0){
        // No need to prompt, just return an empty String array
        return new String[0];
      }
      try{
        if(attemptCount==0&&password!=null&&prompt.length==1
            &&prompt[0].trim().equalsIgnoreCase("password:")){ //$NON-NLS-1$
          // Return the provided password the first time but always prompt on subsequent tries
          attemptCount++;
          return new String[] {password};
        }
        String[] result=authenticator.promptForKeyboradInteractive(location,
            destination, name, instruction, prompt, echo);
        if(result==null)
          return null; // canceled
        if(result.length==1&&prompt.length==1
            &&prompt[0].trim().equalsIgnoreCase("password:")){ //$NON-NLS-1$
          password=result[0];
          passwordChanged=true;
        }
        attemptCount++;
        return result;
      }
      catch(OperationCanceledException e){
        return null;
      }
      catch(JSchCoreException e){
        return null;
      }
    }

    /**
     * Callback to indicate that a connection is about to be attempted
     */
    public void aboutToConnect(){
      attemptCount=0;
      passwordChanged=false;
    }

    /**
     * Callback to indicate that a connection was made
     */
    public void connectionMade(){
      attemptCount=0;
      if(passwordChanged&&password!=null&&location!=null){
        // We were prompted for and returned a password so record it with the location
        location.setPassword(password);
      }
    }
  }

  public static JSchSession getSession(IJSchLocation location,
      IProgressMonitor monitor) throws JSchException{
    return getSession(location, null, monitor, getDefaultContext().getPool());
  }

  public static JSchSession getSession(IJSchLocation location,
      IUserAuthenticator authenticator,
      IProgressMonitor monitor, Hashtable pool) throws JSchException{
    
    String username=location.getUsername();
    String password=location.getPassword();
    String hostname=location.getHost();
    int port=location.getPort();

    if(port==IJSchLocation.USE_DEFAULT_PORT)
      port=getPort(location);

    String key=getPoolKey(username, hostname, port);
    
    try{
      JSchSession jschSession=null;
      if(pool!=null){
        jschSession=(JSchSession)pool.get(key);
      }
      if(jschSession!=null&&!jschSession.getSession().isConnected()){
        if(pool!=null)
        pool.remove(key);
        jschSession=null;
      }

      if(jschSession==null){
        MyUserInfo ui=new MyUserInfo(username, password, location, authenticator);
        UserInfoTimer wrapperUI=new UserInfoTimer(ui);
        ui.aboutToConnect();

        Session session=null;
        try{
          session=createSession(username, password, hostname, port,
              new JSchSession.ResponsiveSocketFacory(monitor), wrapperUI);
        }
        catch(JSchException e){
          if(isAuthenticationFailure(e)&&wrapperUI.hasPromptExceededTimeout()){
            // Try again since the previous prompt may have obtained the proper credentials from the user
            session=createSession(username, password, hostname, port,
                new JSchSession.ResponsiveSocketFacory(monitor),
                wrapperUI);
          }
          else{
            throw e;
          }
        }
        ui.connectionMade();
        JSchSession schSession=new JSchSession(session, location, wrapperUI);
        if(pool!=null)
        pool.put(key, schSession);
        return schSession;
      }
      else{
        return jschSession;
      }
    }
    catch(JSchException e){
      if(pool!=null)
      pool.remove(key);
      if(e.toString().indexOf("Auth cancel")!=-1){ //$NON-NLS-1$
        throw new OperationCanceledException();
      }
      throw e;
    }
  }

  private static Session createSession(String username, String password,
      String hostname, int port, SocketFactory socketFactory,
      com.jcraft.jsch.UserInfo wrapperUI) throws JSchException{
    IJSchService service = JSchCorePlugin.getPlugin().getJSchService();
    if (service == null)
            return null;
    Session session = service.createSession(hostname, port, username);

    if(password!=null)
      session.setPassword(password);
 
    session.setTimeout(getCVSTimeoutInMillis());

    session.setUserInfo(wrapperUI);
    session.setSocketFactory(socketFactory);
    
    Hashtable config=new Hashtable(); 
    config.put("PreferredAuthentications", //$NON-NLS-1$ 
    "gssapi-with-mic,publickey,password,keyboard-interactive"); //$NON-NLS-1$ 
    session.setConfig(config); 

    // This is where the server is contacted and authentication occurs
    try{
      session.connect();
    }
    catch(JSchException e){
      if(session.isConnected())
        session.disconnect();
      throw e;
    }
    return session;
  }

  private static String getPoolKey(String username, String hostname, int port){
    return username+"@"+hostname+":"+port; //$NON-NLS-1$ //$NON-NLS-2$
  }

  private static String getPoolKey(IJSchLocation location){
    return location.getUsername()+"@"+location.getHost()+":"+getPort(location); //$NON-NLS-1$ //$NON-NLS-2$
  }

  private static int getPort(IJSchLocation location){
    int port=location.getPort();
    if(port==IJSchLocation.USE_DEFAULT_PORT)
      port=SSH_DEFAULT_PORT;
    return port;
  }

  public static void shutdown(JSchContext context){
    if(context.getJSch()!=null && context.getPool()!=null && 
        context.getPool().size()>0){
      Hashtable pool=context.getPool();
      for(Enumeration e=pool.elements(); e.hasMoreElements();){
        JSchSession session=(JSchSession)(e.nextElement());
        try{
          session.getSession().disconnect();
        }
        catch(Exception ee){
        }
      }
      pool.clear();
    }
  }

  private JSchSession(Session session, IJSchLocation location,
      com.jcraft.jsch.UserInfo prompter){
    this.session=session;
    this.location=location;
    this.prompter=prompter;
  }

  public Session getSession(){
    return session;
  }

  public com.jcraft.jsch.UserInfo getPrompter(){
    return prompter;
  }

  public boolean hasPromptExceededTimeout(){
    if(prompter instanceof UserInfoTimer){
      UserInfoTimer timer=(UserInfoTimer)prompter;
      if(!timer.isPrompting()){
        return timer.getLastDuration()>getCVSTimeoutInMillis();
      }
    }
    return false;
  }

  public void dispose(JSchContext context){
    if(session.isConnected()){
      session.disconnect();
    }
    Hashtable pool=context.getPool();
    if(pool!=null)
    pool.remove(getPoolKey(location));
  }

  public static boolean isAuthenticationFailure(JSchException ee){
    return ee.getMessage().equals("Auth fail"); //$NON-NLS-1$
  }

  public boolean isChannelNotOpenError(JSchException ee){
    return ee.getMessage().indexOf("channel is not opened")!=-1; //$NON-NLS-1$
  }

  public boolean isSessionDownError(JSchException ee){
    return ee.getMessage().equals("session is down"); //$NON-NLS-1$
  }

  public boolean isSSH2Unsupported(JSchException e){
    return e.toString().indexOf("invalid server's version string")!=-1; //$NON-NLS-1$
  }
}
