/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.jcraft.eclipse.jsch.core;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;


/**
 * A checked expection representing a failure in the CVS plugin.
 * <p>
 * CVS exceptions contain a status object describing the cause of 
 * the exception.
 * </p>
 *
 * @see IStatus
 */
public class JSchCoreException extends CoreException{

  // Field required to avoid compiler warning
  private static final long serialVersionUID=1L;

  // The operation completed successfully.
  public static final int OK=0;

  // The operation failed because the resource is not checked-in.
  public static final int NOT_CHECKED_IN=-1;

  // The operation failed because the resource is not checked-out.
  public static final int NOT_CHECKED_OUT=-2;

  // The corresponding remote resource no longer exists or was never created.
  public static final int NO_REMOTE_RESOURCE=-3;

  // The provider suffered an IO failure, the operation may be retried.
  public static final int IO_FAILED=-4;

  // The user is not authorized to execute the attempted operation.
  public static final int NOT_AUTHORIZED=-5;

  // The provider was unable to complete the operation for an unspecified reason.
  public static final int UNABLE=-6;

  // The operation cannot be performed due to a conflict with other work.
  public static final int CONFLICT=-7;

  public JSchCoreException(CoreException e){
    super(asStatus(e));//super(e);
  }

  private static Status asStatus(CoreException e){
    IStatus status=e.getStatus();
    return new Status(status.getSeverity(), status.getPlugin(), status
        .getCode(), status.getMessage(), e);
  }

  public JSchCoreException(int severity, int code, String message, Throwable e){
    super(new JSchCoreStatus(severity, code, message, e));
  }

  public JSchCoreException(int severity, int code, String message){
    this(severity, code, message, null);
  }

  public JSchCoreException(String message){
    super(new JSchCoreStatus(IStatus.ERROR, UNABLE, message, null));
  }

  public JSchCoreException(String message, Throwable e){
    this(IStatus.ERROR, UNABLE, message, e);
  }

  public JSchCoreException(IStatus status){
    super(status);
  }

  /*
   * Static helper methods for creating exceptions
   */
  public static JSchCoreException wrapException(IResource resource, String message,
      IOException e){
    // NOTE: we should record the resource somehow
    // We should also inlcude the IO message
    return new JSchCoreException(new JSchCoreStatus(IStatus.ERROR, IO_FAILED, message, e));
  }

  /*
   * Static helper methods for creating exceptions
   */
  public static JSchCoreException wrapException(IResource resource, String message,
      CoreException e){
    return new JSchCoreException(new JSchCoreStatus(IStatus.ERROR, e.getStatus()
        .getCode(), message, e));
  }

  /*
   * Static helper methods for creating exceptions
   */
  public static JSchCoreException wrapException(Exception e){
    Throwable t=e;
    if(e instanceof InvocationTargetException){
      Throwable target=((InvocationTargetException)e).getTargetException();
      if(target instanceof JSchCoreException){
        return (JSchCoreException)target;
      }
      t=target;
    }
    return new JSchCoreException(new JSchCoreStatus(IStatus.ERROR, UNABLE,
        t.getMessage()!=null ? t.getMessage() : "", t)); //$NON-NLS-1$
  }

  public static JSchCoreException wrapException(CoreException e){
    if(e instanceof JSchCoreException){
      return (JSchCoreException)e;
    }
    return new JSchCoreException(e);
  }
}
